// Portal de Cursos - Complete Database Schema
// Following Replit Auth and PostgreSQL blueprints
import { sql } from 'drizzle-orm';
import { relations } from 'drizzle-orm';
import {
  index,
  jsonb,
  pgTable,
  timestamp,
  varchar,
  text,
  integer,
  boolean,
  decimal,
} from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// ============ AUTH TABLES (From Replit Auth Blueprint) ============

// Session storage table (mandatory for Replit Auth)
export const sessions = pgTable(
  "sessions",
  {
    sid: varchar("sid").primaryKey(),
    sess: jsonb("sess").notNull(),
    expire: timestamp("expire").notNull(),
  },
  (table) => [index("IDX_session_expire").on(table.expire)],
);

// Users table with role field for admin/student distinction
export const users = pgTable("users", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  email: varchar("email").unique().notNull(),
  password: varchar("password", { length: 255 }), // bcrypt hash (nullable for migration)
  firstName: varchar("first_name"),
  lastName: varchar("last_name"),
  profileImageUrl: varchar("profile_image_url"),
  role: varchar("role", { enum: ["admin", "student"] }).default("student").notNull(),
  points: integer("points").default(0).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export type UpsertUser = typeof users.$inferInsert;
export type User = typeof users.$inferSelect;

// ============ COURSE MANAGEMENT ============

export const courses = pgTable("courses", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  title: varchar("title", { length: 255 }).notNull(),
  description: text("description"),
  thumbnail: text("thumbnail"), // Image stored as base64 text
  thumbnailMimeType: varchar("thumbnail_mime_type", { length: 100 }),
  thumbnailUrl: varchar("thumbnail_url", { length: 1024 }),
  thumbVisaoAluno: varchar("thumb_visao_aluno", { length: 1024 }), // Thumbnail URL para visão do aluno (550x550)
  instructorName: varchar("instructor_name", { length: 255 }),
  instructorBio: text("instructor_bio"),
  status: varchar("status", { enum: ["draft", "published", "archived"] }).default("draft").notNull(),
  category: varchar("category", { length: 120 }),
  totalStudents: integer("total_students").default(0).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const courseShowcases = pgTable("course_showcases", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  name: varchar("name", { length: 255 }).notNull(),
  description: text("description"),
  orderIndex: integer("order_index").default(0).notNull(),
  status: varchar("status", { enum: ["listed", "unlisted"] }).default("listed").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const courseShowcaseCourses = pgTable("course_showcase_courses", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  showcaseId: varchar("showcase_id").notNull().references(() => courseShowcases.id, { onDelete: "cascade" }),
  courseId: varchar("course_id").notNull().references(() => courses.id, { onDelete: "cascade" }),
  orderIndex: integer("order_index").default(0).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
});

export const coursesRelations = relations(courses, ({ many }) => ({
  modules: many(courseModules),
  materials: many(courseMaterials),
  userAccess: many(userCourseAccess),
  showcases: many(courseShowcaseCourses),
}));

export const courseShowcasesRelations = relations(courseShowcases, ({ many }) => ({
  showcaseCourses: many(courseShowcaseCourses),
}));

export const courseShowcaseCoursesRelations = relations(courseShowcaseCourses, ({ one }) => ({
  showcase: one(courseShowcases, {
    fields: [courseShowcaseCourses.showcaseId],
    references: [courseShowcases.id],
  }),
  course: one(courses, {
    fields: [courseShowcaseCourses.courseId],
    references: [courses.id],
  }),
}));

export type Course = typeof courses.$inferSelect;
export const insertCourseSchema = createInsertSchema(courses).omit({ id: true, createdAt: true, updatedAt: true, totalStudents: true });
export type InsertCourse = z.infer<typeof insertCourseSchema>;

export type CourseShowcase = typeof courseShowcases.$inferSelect;
export const insertCourseShowcaseSchema = createInsertSchema(courseShowcases)
  .omit({ id: true, createdAt: true })
  .extend({
    orderIndex: z.number().int().nonnegative().optional(),
    status: z.enum(["listed", "unlisted"]).optional(),
  });
export type InsertCourseShowcase = z.infer<typeof insertCourseShowcaseSchema>;

export type CourseShowcaseCourse = typeof courseShowcaseCourses.$inferSelect;
export const insertCourseShowcaseCourseSchema = createInsertSchema(courseShowcaseCourses)
  .omit({ id: true, createdAt: true })
  .extend({ orderIndex: z.number().int().nonnegative().optional() });
export type InsertCourseShowcaseCourse = z.infer<typeof insertCourseShowcaseCourseSchema>;

// Course Modules (e.g., "Módulo 1: Introdução")
export const courseModules = pgTable("course_modules", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  courseId: varchar("course_id").notNull().references(() => courses.id, { onDelete: "cascade" }),
  title: varchar("title", { length: 255 }).notNull(),
  description: text("description"),
  thumbnailUrl: text("thumbnail_url"),
  orderIndex: integer("order_index").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
});

export const courseModulesRelations = relations(courseModules, ({ one, many }) => ({
  course: one(courses, {
    fields: [courseModules.courseId],
    references: [courses.id],
  }),
  lessons: many(courseLessons),
}));

export type CourseModule = typeof courseModules.$inferSelect;
export const insertCourseModuleSchema = createInsertSchema(courseModules).omit({ id: true, createdAt: true });
export type InsertCourseModule = z.infer<typeof insertCourseModuleSchema>;

// Course Lessons (individual videos/aulas within modules)
export const courseLessons = pgTable("course_lessons", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  moduleId: varchar("module_id").notNull().references(() => courseModules.id, { onDelete: "cascade" }),
  title: varchar("title", { length: 255 }).notNull(),
  description: text("description"),
  thumbnailUrl: varchar("thumbnail_url", { length: 1024 }),
  videoUrl: varchar("video_url", { length: 2048 }),
  videoData: text("video_data"), // Video stored as base64 text
  videoMimeType: varchar("video_mime_type", { length: 100 }),
  videoDuration: integer("video_duration"), // Duration in seconds
  status: varchar("status", { length: 20 }).default("published").notNull(),
  orderIndex: integer("order_index").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export const courseLessonsRelations = relations(courseLessons, ({ one, many }) => ({
  module: one(courseModules, {
    fields: [courseLessons.moduleId],
    references: [courseModules.id],
  }),
  progress: many(studentProgress),
}));

export type CourseLesson = typeof courseLessons.$inferSelect;
export const insertCourseLessonSchema = createInsertSchema(courseLessons)
  .omit({ id: true, createdAt: true, updatedAt: true })
  .extend({
    status: z.enum(["published", "unlisted"]).optional(),
    thumbnailUrl: z.string().url().optional(),
    videoUrl: z.string().url().optional(),
  });
export type InsertCourseLesson = z.infer<typeof insertCourseLessonSchema>;

// Course Materials (PDFs, documents stored as base64 text)
export const courseMaterials = pgTable("course_materials", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  courseId: varchar("course_id").references(() => courses.id, { onDelete: "cascade" }),
  lessonId: varchar("lesson_id").references(() => courseLessons.id, { onDelete: "cascade" }),
  title: varchar("title", { length: 255 }).notNull(),
  description: text("description"),
  filePath: text("file_path").notNull(),
  fileName: varchar("file_name", { length: 255 }).notNull(),
  fileMimeType: varchar("file_mime_type", { length: 100 }).notNull(),
  fileSize: integer("file_size"), // Size in bytes
  materialType: varchar("material_type", { enum: ["pdf", "document", "other"] }).default("other").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
});

export const courseMaterialsRelations = relations(courseMaterials, ({ one }) => ({
  course: one(courses, {
    fields: [courseMaterials.courseId],
    references: [courses.id],
  }),
  lesson: one(courseLessons, {
    fields: [courseMaterials.lessonId],
    references: [courseLessons.id],
  }),
}));

export type CourseMaterial = typeof courseMaterials.$inferSelect;
export const insertCourseMaterialSchema = createInsertSchema(courseMaterials)
  .omit({ id: true, createdAt: true })
  .extend({
    courseId: z.string().optional(),
    lessonId: z.string().optional(),
  });
export type InsertCourseMaterial = z.infer<typeof insertCourseMaterialSchema>;

// ============ STUDENT TRACKING ============

// User Course Access (which users can access which courses)
export const userCourseAccess = pgTable("user_course_access", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  courseId: varchar("course_id").notNull().references(() => courses.id, { onDelete: "cascade" }),
  accessGrantedAt: timestamp("access_granted_at").defaultNow(),
  lastAccessedAt: timestamp("last_accessed_at"),
});

export const userCourseAccessRelations = relations(userCourseAccess, ({ one }) => ({
  user: one(users, {
    fields: [userCourseAccess.userId],
    references: [users.id],
  }),
  course: one(courses, {
    fields: [userCourseAccess.courseId],
    references: [courses.id],
  }),
}));

export type UserCourseAccess = typeof userCourseAccess.$inferSelect;
export const insertUserCourseAccessSchema = createInsertSchema(userCourseAccess).omit({ id: true, accessGrantedAt: true });
export type InsertUserCourseAccess = z.infer<typeof insertUserCourseAccessSchema>;

// Student Progress (track completed lessons)
export const studentProgress = pgTable("student_progress", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  lessonId: varchar("lesson_id").notNull().references(() => courseLessons.id, { onDelete: "cascade" }),
  completed: boolean("completed").default(false).notNull(),
  progressPercentage: integer("progress_percentage").default(0).notNull(),
  completedAt: timestamp("completed_at"),
  lastWatchedAt: timestamp("last_watched_at").defaultNow(),
});

export const studentProgressRelations = relations(studentProgress, ({ one }) => ({
  user: one(users, {
    fields: [studentProgress.userId],
    references: [users.id],
  }),
  lesson: one(courseLessons, {
    fields: [studentProgress.lessonId],
    references: [courseLessons.id],
  }),
}));

export type StudentProgress = typeof studentProgress.$inferSelect;
export const insertStudentProgressSchema = createInsertSchema(studentProgress).omit({ id: true, lastWatchedAt: true });
export type InsertStudentProgress = z.infer<typeof insertStudentProgressSchema>;

// Certificates Issued (for dashboard metrics)
export const certificates = pgTable("certificates", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  courseId: varchar("course_id").notNull().references(() => courses.id, { onDelete: "cascade" }),
  issuedAt: timestamp("issued_at").defaultNow(),
  certificateData: text("certificate_data"), // PDF certificate stored as base64 text
});

export type Certificate = typeof certificates.$inferSelect;
export const insertCertificateSchema = createInsertSchema(certificates).omit({ id: true, issuedAt: true });
export type InsertCertificate = z.infer<typeof insertCertificateSchema>;

// Student Access Logs (for dashboard access metrics)
export const accessLogs = pgTable("access_logs", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  accessedAt: timestamp("accessed_at").defaultNow().notNull(),
});

export type AccessLog = typeof accessLogs.$inferSelect;
export const insertAccessLogSchema = createInsertSchema(accessLogs).omit({ id: true });
export type InsertAccessLog = z.infer<typeof insertAccessLogSchema>;

// ============ SALES & WEBHOOKS ============

export const sales = pgTable("sales", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  webhookId: varchar("webhook_id").unique(), // ID from external platform
  courseId: varchar("course_id").references(() => courses.id),
  buyerEmail: varchar("buyer_email", { length: 255 }).notNull(),
  status: varchar("status", { enum: ["processed", "pending", "error"] }).default("pending").notNull(),
  amount: decimal("amount", { precision: 10, scale: 2 }),
  currency: varchar("currency", { length: 10 }).default("BRL"),
  purchasedAt: timestamp("purchased_at").notNull(),
  processedAt: timestamp("processed_at"),
  createdAt: timestamp("created_at").defaultNow(),
});

export type Sale = typeof sales.$inferSelect;
export const insertSaleSchema = createInsertSchema(sales).omit({ id: true, createdAt: true });
export type InsertSale = z.infer<typeof insertSaleSchema>;

// ============ SUPPORT & FAQ ============

export const supportTickets = pgTable("support_tickets", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  userId: varchar("user_id").references(() => users.id, { onDelete: "set null" }),
  userEmail: varchar("user_email", { length: 255 }).notNull(),
  subject: varchar("subject", { length: 255 }).notNull(),
  message: text("message").notNull(),
  status: varchar("status", { enum: ["open", "in_progress", "resolved", "closed"] }).default("open").notNull(),
  priority: varchar("priority", { enum: ["low", "medium", "high", "urgent"] }).default("medium").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export type SupportTicket = typeof supportTickets.$inferSelect;
export const insertSupportTicketSchema = createInsertSchema(supportTickets).omit({ id: true, createdAt: true, updatedAt: true });
export type InsertSupportTicket = z.infer<typeof insertSupportTicketSchema>;

export const faqItems = pgTable("faq_items", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  question: varchar("question", { length: 500 }).notNull(),
  answer: text("answer").notNull(),
  category: varchar("category", { length: 100 }),
  orderIndex: integer("order_index").default(0).notNull(),
  isPublished: boolean("is_published").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

export type FaqItem = typeof faqItems.$inferSelect;
export const insertFaqItemSchema = createInsertSchema(faqItems).omit({ id: true, createdAt: true, updatedAt: true });
export type InsertFaqItem = z.infer<typeof insertFaqItemSchema>;
