// Database Storage - Complete implementation for Portal de Cursos
import {
  users,
  courses,
  courseModules,
  courseLessons,
  courseMaterials,
  userCourseAccess,
  studentProgress,
  certificates,
  accessLogs,
  sales,
  supportTickets,
  faqItems,
  courseShowcases,
  courseShowcaseCourses,
  type User,
  type UpsertUser,
  type Course,
  type InsertCourse,
  type CourseModule,
  type InsertCourseModule,
  type CourseLesson,
  type InsertCourseLesson,
  type CourseMaterial,
  type InsertCourseMaterial,
  type UserCourseAccess,
  type InsertUserCourseAccess,
  type StudentProgress,
  type InsertStudentProgress,
  type Sale,
  type InsertSale,
  type SupportTicket,
  type InsertSupportTicket,
  type FaqItem,
  type InsertFaqItem,
  type CourseShowcase,
  type InsertCourseShowcase,
  type CourseShowcaseCourse,
  type InsertCourseShowcaseCourse,
} from "@shared/schema";
import { db } from "./db";
import { eq, desc, and, gte, lt, sql, isNull, inArray } from "drizzle-orm";

// Interface for storage operations
export type CourseShowcaseWithCourses = CourseShowcase & { courses: (Course & { lessonsCount?: number })[] };

export interface IStorage {
  // User operations (mandatory for Replit Auth)
  getUser(id: string): Promise<User | undefined>;
  upsertUser(user: UpsertUser): Promise<User>;
  getAllUsers(): Promise<User[]>;
  
  // Course operations
  getAllCourses(): Promise<Course[]>;
  getCourse(id: string): Promise<Course | undefined>;
  createCourse(course: InsertCourse): Promise<Course>;
  updateCourse(id: string, course: Partial<InsertCourse>): Promise<Course>;
  deleteCourse(id: string): Promise<void>;

  // Course showcases (vitrines)
  getAllCourseShowcases(): Promise<CourseShowcaseWithCourses[]>;
  createCourseShowcase(showcase: InsertCourseShowcase): Promise<CourseShowcase>;
  deleteCourseShowcase(id: string): Promise<void>;
  addCourseToShowcase(link: InsertCourseShowcaseCourse): Promise<CourseShowcaseCourse>;
  removeCourseFromShowcase(showcaseId: string, courseId: string): Promise<void>;
  reorderShowcaseCourses(showcaseId: string, courseIds: string[]): Promise<void>;
  reorderCourseShowcases(showcaseIds: string[]): Promise<void>;
  updateCourseShowcase(id: string, data: Partial<InsertCourseShowcase>): Promise<CourseShowcase>;
  
  // Course modules
  getAllModules(): Promise<CourseModule[]>;
  getModulesByCourse(courseId: string): Promise<CourseModule[]>;
  getModule(id: string): Promise<CourseModule | undefined>;
  createModule(module: InsertCourseModule): Promise<CourseModule>;
  deleteModule(moduleId: string): Promise<void>;
  
  // Course lessons
  getLessonsByModule(moduleId: string): Promise<CourseLesson[]>;
  getLesson(id: string): Promise<CourseLesson | undefined>;
  createLesson(lesson: InsertCourseLesson): Promise<CourseLesson>;
  updateLesson(id: string, lesson: Partial<InsertCourseLesson>): Promise<CourseLesson>;
  updateLessonStatus(id: string, status: string): Promise<CourseLesson>;
  deleteLesson(id: string): Promise<void>;
  
  // Course materials
  getMaterialsByCourse(courseId: string): Promise<CourseMaterial[]>;
  getMaterialsByLesson(lessonId: string): Promise<CourseMaterial[]>;
  getMaterial(id: string): Promise<CourseMaterial | undefined>;
  createMaterial(material: InsertCourseMaterial): Promise<CourseMaterial>;
  deleteMaterial(id: string): Promise<void>;
  
  // User course access
  getUserCourseAccess(userId: string): Promise<UserCourseAccess[]>;
  grantCourseAccess(access: InsertUserCourseAccess): Promise<UserCourseAccess>;
  
  // Student progress
  getStudentProgress(userId: string, lessonId: string): Promise<StudentProgress | undefined>;
  upsertStudentProgress(progress: InsertStudentProgress): Promise<StudentProgress>;
  getCourseProgress(userId: string, courseId: string): Promise<StudentProgress[]>;
  
  // Access logs
  logAccess(userId: string): Promise<void>;
  
  // Sales
  getAllSales(): Promise<Sale[]>;
  createSale(sale: InsertSale): Promise<Sale>;
  
  // Support
  getAllTickets(): Promise<SupportTicket[]>;
  createTicket(ticket: InsertSupportTicket): Promise<SupportTicket>;
  
  // FAQ
  getAllFaqs(): Promise<FaqItem[]>;
  createFaq(faq: InsertFaqItem): Promise<FaqItem>;
  
  // Dashboard metrics
  getDashboardMetrics(): Promise<any>;
}

export class DatabaseStorage implements IStorage {
  // ============ USER OPERATIONS ============
  async getUser(id: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user;
  }

  async upsertUser(userData: UpsertUser): Promise<User> {
    const [user] = await db
      .insert(users)
      .values(userData)
      .onConflictDoUpdate({
        target: users.id,
        set: {
          ...userData,
          updatedAt: new Date(),
        },
      })
      .returning();
    return user;
  }

  async getAllUsers(): Promise<User[]> {
    return await db.select().from(users).orderBy(desc(users.createdAt));
  }

  // ============ COURSE OPERATIONS ============
  async getAllCourses(): Promise<Course[]> {
    return await db.select().from(courses).orderBy(desc(courses.createdAt));
  }

  async getCourse(id: string): Promise<Course | undefined> {
    const [course] = await db.select().from(courses).where(eq(courses.id, id));
    return course;
  }

  async createCourse(courseData: InsertCourse): Promise<Course> {
    const [course] = await db.insert(courses).values(courseData).returning();
    return course;
  }

  async updateCourse(id: string, courseData: Partial<InsertCourse>): Promise<Course> {
    const [course] = await db
      .update(courses)
      .set({ ...courseData, updatedAt: new Date() })
      .where(eq(courses.id, id))
      .returning();
    return course;
  }

  async deleteCourse(id: string): Promise<void> {
    await db.delete(courses).where(eq(courses.id, id));
  }

  // ============ COURSE SHOWCASE OPERATIONS ============
  async getAllCourseShowcases(): Promise<CourseShowcaseWithCourses[]> {
    const showcases = await db
      .select()
      .from(courseShowcases)
      .orderBy(courseShowcases.orderIndex, desc(courseShowcases.createdAt));

    if (showcases.length === 0) {
      return [];
    }

    const showcaseIds = showcases.map((showcase) => showcase.id);
    const showcaseCoursesRows = await db
      .select({
        id: courseShowcaseCourses.id,
        showcaseId: courseShowcaseCourses.showcaseId,
        course: courses,
        orderIndex: courseShowcaseCourses.orderIndex,
      })
      .from(courseShowcaseCourses)
      .leftJoin(courses, eq(courseShowcaseCourses.courseId, courses.id))
      .where(inArray(courseShowcaseCourses.showcaseId, showcaseIds))
      .orderBy(
        courseShowcaseCourses.showcaseId,
        courseShowcaseCourses.orderIndex,
        desc(courses.createdAt),
      );

    const courseIds = showcaseCoursesRows
      .map((row) => row.course?.id)
      .filter((id): id is string => Boolean(id));

    const lessonsCountByCourseId = new Map<string, number>();

    if (courseIds.length > 0) {
      const lessonCounts = await db
        .select({
          courseId: courseModules.courseId,
          count: sql<number>`COALESCE(COUNT(${courseLessons.id}), 0)::int`,
        })
        .from(courseModules)
        .leftJoin(courseLessons, eq(courseLessons.moduleId, courseModules.id))
        .where(inArray(courseModules.courseId, courseIds))
        .groupBy(courseModules.courseId);

      for (const row of lessonCounts) {
        lessonsCountByCourseId.set(row.courseId, row.count ?? 0);
      }
    }

    const coursesByShowcase = new Map<string, (Course & { lessonsCount?: number })[]>();
    for (const row of showcaseCoursesRows) {
      if (!row.course) continue;
      const list = coursesByShowcase.get(row.showcaseId) ?? [];
      list.push({
        ...row.course,
        lessonsCount: lessonsCountByCourseId.get(row.course.id) ?? 0,
      });
      coursesByShowcase.set(row.showcaseId, list);
    }

    return showcases.map((showcase) => ({
      ...showcase,
      courses: coursesByShowcase.get(showcase.id) ?? [],
    }));
  }

  async createCourseShowcase(showcaseData: InsertCourseShowcase): Promise<CourseShowcase> {
    const [largestOrder] = await db
      .select({
        maxOrder: sql<number>`COALESCE(MAX(${courseShowcases.orderIndex}), -1)`,
      })
      .from(courseShowcases);

    const orderIndex =
      showcaseData.orderIndex ??
      (largestOrder?.maxOrder !== undefined ? largestOrder.maxOrder + 1 : 0);

    const { status, ...rest } = showcaseData;

    const [showcase] = await db
      .insert(courseShowcases)
      .values({ ...rest, orderIndex, status: status ?? "listed" })
      .returning();
    return showcase;
  }

  async deleteCourseShowcase(id: string): Promise<void> {
    await db.delete(courseShowcases).where(eq(courseShowcases.id, id));
  }

  async addCourseToShowcase(link: InsertCourseShowcaseCourse): Promise<CourseShowcaseCourse> {
    const [largestOrder] = await db
      .select({
        maxOrder: sql<number>`COALESCE(MAX(${courseShowcaseCourses.orderIndex}), -1)`,
      })
      .from(courseShowcaseCourses)
      .where(eq(courseShowcaseCourses.showcaseId, link.showcaseId));

    const orderIndex =
      link.orderIndex ?? (largestOrder?.maxOrder !== undefined ? largestOrder.maxOrder + 1 : 0);

    const [record] = await db
      .insert(courseShowcaseCourses)
      .values({ ...link, orderIndex })
      .returning();
    return record;
  }

  async removeCourseFromShowcase(showcaseId: string, courseId: string): Promise<void> {
    await db
      .delete(courseShowcaseCourses)
      .where(and(eq(courseShowcaseCourses.showcaseId, showcaseId), eq(courseShowcaseCourses.courseId, courseId)));
  }

  async reorderShowcaseCourses(showcaseId: string, courseIds: string[]): Promise<void> {
    await db.transaction(async (tx) => {
      for (let index = 0; index < courseIds.length; index++) {
        const courseId = courseIds[index];
        await tx
          .update(courseShowcaseCourses)
          .set({ orderIndex: index })
          .where(and(eq(courseShowcaseCourses.showcaseId, showcaseId), eq(courseShowcaseCourses.courseId, courseId)));
      }
    });
  }

  async reorderCourseShowcases(showcaseIds: string[]): Promise<void> {
    await db.transaction(async (tx) => {
      for (let index = 0; index < showcaseIds.length; index++) {
        await tx
          .update(courseShowcases)
          .set({ orderIndex: index })
          .where(eq(courseShowcases.id, showcaseIds[index]));
      }
    });
  }

  async updateCourseShowcase(id: string, data: Partial<InsertCourseShowcase>): Promise<CourseShowcase> {
    const [updated] = await db
      .update(courseShowcases)
      .set({ ...data, updatedAt: new Date() })
      .where(eq(courseShowcases.id, id))
      .returning();
    return updated;
  }

  // ============ MODULE OPERATIONS ============
  async getAllModules(): Promise<CourseModule[]> {
    return await db.select().from(courseModules).orderBy(courseModules.orderIndex);
  }

  async getModulesByCourse(courseId: string): Promise<CourseModule[]> {
    return await db
      .select()
      .from(courseModules)
      .where(eq(courseModules.courseId, courseId))
      .orderBy(courseModules.orderIndex);
  }

  async getModule(id: string): Promise<CourseModule | undefined> {
    const [module] = await db.select().from(courseModules).where(eq(courseModules.id, id));
    return module;
  }

  async createModule(moduleData: InsertCourseModule): Promise<CourseModule> {
    const [module] = await db.insert(courseModules).values(moduleData).returning();
    return module;
  }

  async deleteModule(moduleId: string): Promise<void> {
    await db.delete(courseModules).where(eq(courseModules.id, moduleId));
  }

  // ============ LESSON OPERATIONS ============
  async getLessonsByModule(moduleId: string): Promise<CourseLesson[]> {
    return await db
      .select()
      .from(courseLessons)
      .where(eq(courseLessons.moduleId, moduleId))
      .orderBy(courseLessons.orderIndex);
  }

  async getLesson(id: string): Promise<CourseLesson | undefined> {
    const [lesson] = await db.select().from(courseLessons).where(eq(courseLessons.id, id));
    return lesson;
  }

  async createLesson(lessonData: InsertCourseLesson): Promise<CourseLesson> {
    const [lesson] = await db
      .insert(courseLessons)
      .values({
        ...lessonData,
        status: lessonData.status ?? "published",
      })
      .returning();
    return lesson;
  }

  async updateLesson(id: string, lessonData: Partial<InsertCourseLesson>): Promise<CourseLesson> {
    const [lesson] = await db
      .update(courseLessons)
      .set({ ...lessonData, updatedAt: new Date() })
      .where(eq(courseLessons.id, id))
      .returning();
    return lesson;
  }

  async updateLessonStatus(id: string, status: string): Promise<CourseLesson> {
    const [lesson] = await db
      .update(courseLessons)
      .set({ status, updatedAt: new Date() })
      .where(eq(courseLessons.id, id))
      .returning();
    return lesson;
  }

  async deleteLesson(id: string): Promise<void> {
    await db.delete(courseLessons).where(eq(courseLessons.id, id));
  }

  // ============ MATERIAL OPERATIONS ============
  async getMaterialsByCourse(courseId: string): Promise<CourseMaterial[]> {
    return await db
      .select()
      .from(courseMaterials)
      .where(and(eq(courseMaterials.courseId, courseId), isNull(courseMaterials.lessonId)));
  }

  async getMaterialsByLesson(lessonId: string): Promise<CourseMaterial[]> {
    return await db
      .select()
      .from(courseMaterials)
      .where(eq(courseMaterials.lessonId, lessonId));
  }

  async getMaterial(id: string): Promise<CourseMaterial | undefined> {
    const [material] = await db.select().from(courseMaterials).where(eq(courseMaterials.id, id));
    return material;
  }

  async createMaterial(materialData: InsertCourseMaterial): Promise<CourseMaterial> {
    if (!materialData.courseId && !materialData.lessonId) {
      throw new Error("courseId or lessonId must be provided");
    }
    const [material] = await db.insert(courseMaterials).values(materialData).returning();
    return material;
  }

  async deleteMaterial(id: string): Promise<void> {
    await db.delete(courseMaterials).where(eq(courseMaterials.id, id));
  }

  // ============ USER COURSE ACCESS ============
  async getUserCourseAccess(userId: string): Promise<UserCourseAccess[]> {
    return await db
      .select()
      .from(userCourseAccess)
      .where(eq(userCourseAccess.userId, userId));
  }

  async grantCourseAccess(accessData: InsertUserCourseAccess): Promise<UserCourseAccess> {
    const [access] = await db.insert(userCourseAccess).values(accessData).returning();
    return access;
  }

  // ============ STUDENT PROGRESS ============
  async getStudentProgress(userId: string, lessonId: string): Promise<StudentProgress | undefined> {
    const [progress] = await db
      .select()
      .from(studentProgress)
      .where(and(eq(studentProgress.userId, userId), eq(studentProgress.lessonId, lessonId)));
    return progress;
  }

  async upsertStudentProgress(progressData: InsertStudentProgress): Promise<StudentProgress> {
    const existing = await this.getStudentProgress(progressData.userId, progressData.lessonId);
    
    if (existing) {
      const [updated] = await db
        .update(studentProgress)
        .set({
          ...progressData,
          completedAt: progressData.completed ? new Date() : existing.completedAt,
        })
        .where(eq(studentProgress.id, existing.id))
        .returning();
      return updated;
    } else {
      const [created] = await db.insert(studentProgress).values(progressData).returning();
      return created;
    }
  }

  async getCourseProgress(userId: string, courseId: string): Promise<StudentProgress[]> {
    // Get all lessons for the course, then get progress for those lessons
    const modules = await this.getModulesByCourse(courseId);
    const lessonIds: string[] = [];
    
    for (const module of modules) {
      const lessons = await this.getLessonsByModule(module.id);
      lessonIds.push(...lessons.map(l => l.id));
    }
    
    if (lessonIds.length === 0) return [];
    
    return await db
      .select()
      .from(studentProgress)
      .where(
        and(
          eq(studentProgress.userId, userId),
          sql`${studentProgress.lessonId} IN ${lessonIds}`
        )
      );
  }

  // ============ ACCESS LOGS ============
  async logAccess(userId: string): Promise<void> {
    await db.insert(accessLogs).values({ userId });
  }

  // ============ SALES ============
  async getAllSales(): Promise<Sale[]> {
    return await db.select().from(sales).orderBy(desc(sales.purchasedAt));
  }

  async createSale(saleData: InsertSale): Promise<Sale> {
    const [sale] = await db.insert(sales).values(saleData).returning();
    return sale;
  }

  // ============ SUPPORT ============
  async getAllTickets(): Promise<SupportTicket[]> {
    return await db.select().from(supportTickets).orderBy(desc(supportTickets.createdAt));
  }

  async createTicket(ticketData: InsertSupportTicket): Promise<SupportTicket> {
    const [ticket] = await db.insert(supportTickets).values(ticketData).returning();
    return ticket;
  }

  // ============ FAQ ============
  async getAllFaqs(): Promise<FaqItem[]> {
    return await db
      .select()
      .from(faqItems)
      .where(eq(faqItems.isPublished, true))
      .orderBy(faqItems.orderIndex);
  }

  async createFaq(faqData: InsertFaqItem): Promise<FaqItem> {
    const [faq] = await db.insert(faqItems).values(faqData).returning();
    return faq;
  }

  // ============ DASHBOARD METRICS ============
  async getDashboardMetrics(): Promise<any> {
    // Total students
    const totalStudents = await db
      .select({ count: sql<number>`count(*)::int` })
      .from(users)
      .where(eq(users.role, "student"));

    // Certificates issued
    const certificatesIssued = await db
      .select({ count: sql<number>`count(*)::int` })
      .from(certificates);

    // Invalid emails (emails that are null)
    const invalidEmails = await db
      .select({ count: sql<number>`count(*)::int` })
      .from(users)
      .where(isNull(users.email));

    // Access last 30 days
    const thirtyDaysAgo = new Date();
    thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
    const accessLast30Days = await db
      .select({ count: sql<number>`count(*)::int` })
      .from(accessLogs)
      .where(gte(accessLogs.accessedAt, thirtyDaysAgo));

    // Students by month (last 12 months)
    const twelveMonthsAgo = new Date();
    twelveMonthsAgo.setMonth(twelveMonthsAgo.getMonth() - 12);
    
    const studentsByMonthRaw = await db
      .select({
        yearMonth: sql<string>`TO_CHAR(${users.createdAt}, 'YYYY-MM')`,
        month: sql<string>`TO_CHAR(${users.createdAt}, 'Mon')`,
        count: sql<number>`count(*)::int`,
      })
      .from(users)
      .where(
        and(
          eq(users.role, "student"),
          gte(users.createdAt, twelveMonthsAgo)
        )
      )
      .groupBy(sql`TO_CHAR(${users.createdAt}, 'YYYY-MM')`, sql`TO_CHAR(${users.createdAt}, 'Mon')`)
      .orderBy(sql`TO_CHAR(${users.createdAt}, 'YYYY-MM')`);
    
    // Keep both yearMonth and month for frontend
    const studentsByMonth = studentsByMonthRaw;

    // Access segmentation (hoje, ontem, 2-7 dias, 7-14 dias, 14-30 dias)
    const now = new Date();
    const startOfToday = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const startOfYesterday = new Date(startOfToday);
    startOfYesterday.setDate(startOfYesterday.getDate() - 1);
    const twoDaysAgo = new Date(startOfToday);
    twoDaysAgo.setDate(twoDaysAgo.getDate() - 2);
    const sevenDaysAgo = new Date(startOfToday);
    sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
    const fourteenDaysAgo = new Date(startOfToday);
    fourteenDaysAgo.setDate(fourteenDaysAgo.getDate() - 14);

    // Count unique users per period
    const todayAccess = await db
      .select({ count: sql<number>`count(DISTINCT ${accessLogs.userId})::int` })
      .from(accessLogs)
      .where(gte(accessLogs.accessedAt, startOfToday));

    const yesterdayAccess = await db
      .select({ count: sql<number>`count(DISTINCT ${accessLogs.userId})::int` })
      .from(accessLogs)
      .where(
        and(
          gte(accessLogs.accessedAt, startOfYesterday),
          lt(accessLogs.accessedAt, startOfToday)
        )
      );

    const twoDaysTo7Access = await db
      .select({ count: sql<number>`count(DISTINCT ${accessLogs.userId})::int` })
      .from(accessLogs)
      .where(
        and(
          gte(accessLogs.accessedAt, sevenDaysAgo),
          lt(accessLogs.accessedAt, twoDaysAgo)
        )
      );

    const sevenTo14Access = await db
      .select({ count: sql<number>`count(DISTINCT ${accessLogs.userId})::int` })
      .from(accessLogs)
      .where(
        and(
          gte(accessLogs.accessedAt, fourteenDaysAgo),
          lt(accessLogs.accessedAt, sevenDaysAgo)
        )
      );

    const fourteenTo30Access = await db
      .select({ count: sql<number>`count(DISTINCT ${accessLogs.userId})::int` })
      .from(accessLogs)
      .where(
        and(
          gte(accessLogs.accessedAt, thirtyDaysAgo),
          lt(accessLogs.accessedAt, fourteenDaysAgo)
        )
      );

    const accessSegmentation = [
      { period: "Hoje", count: todayAccess[0]?.count || 0 },
      { period: "Ontem", count: yesterdayAccess[0]?.count || 0 },
      { period: "2-7 dias", count: twoDaysTo7Access[0]?.count || 0 },
      { period: "7-14 dias", count: sevenTo14Access[0]?.count || 0 },
      { period: "14-30 dias", count: fourteenTo30Access[0]?.count || 0 },
    ];

    // Points ranking
    const pointsRanking = await db
      .select({
        userId: users.id,
        name: sql<string>`${users.firstName} || ' ' || ${users.lastName}`,
        email: users.email,
        points: users.points,
        profileImageUrl: users.profileImageUrl,
      })
      .from(users)
      .where(eq(users.role, "student"))
      .orderBy(desc(users.points))
      .limit(10);

    // Engagement ranking - real data from student_progress
    const engagementData = await db
      .select({
        userId: studentProgress.userId,
        lessonId: studentProgress.lessonId,
        progressPercentage: studentProgress.progressPercentage,
        lastActivity: studentProgress.lastWatchedAt,
      })
      .from(studentProgress)
      .orderBy(desc(studentProgress.lastWatchedAt))
      .limit(50);

    // Group by user and calculate average progress
    const userEngagement = new Map<string, { 
      userId: string; 
      totalProgress: number; 
      lessonCount: number; 
      lastActivity: Date;
      courseIds: Set<string>;
    }>();

    for (const record of engagementData) {
      if (!userEngagement.has(record.userId)) {
        userEngagement.set(record.userId, {
          userId: record.userId,
          totalProgress: 0,
          lessonCount: 0,
          lastActivity: record.lastActivity || new Date(),
          courseIds: new Set(),
        });
      }
      const userData = userEngagement.get(record.userId)!;
      userData.totalProgress += record.progressPercentage || 0;
      userData.lessonCount += 1;
      
      // Get course ID from lesson
      const lesson = await db
        .select({ moduleId: courseLessons.moduleId })
        .from(courseLessons)
        .where(eq(courseLessons.id, record.lessonId))
        .limit(1);
      
      if (lesson[0]) {
        const module = await db
          .select({ courseId: courseModules.courseId })
          .from(courseModules)
          .where(eq(courseModules.id, lesson[0].moduleId))
          .limit(1);
        
        if (module[0]) {
          userData.courseIds.add(module[0].courseId);
        }
      }

      if (record.lastActivity && record.lastActivity > userData.lastActivity) {
        userData.lastActivity = record.lastActivity;
      }
    }

    // Build engagement ranking
    const engagementRanking = [];
    for (const [userId, data] of Array.from(userEngagement.entries())) {
      const user = await db
        .select({
          firstName: users.firstName,
          lastName: users.lastName,
        })
        .from(users)
        .where(eq(users.id, userId))
        .limit(1);

      if (user[0]) {
        // Get course name (use first course from set)
        const courseId = Array.from(data.courseIds)[0];
        let courseName = "Curso";
        
        if (courseId) {
          const courseResult = await db
            .select({ title: courses.title })
            .from(courses)
            .where(sql`${courses.id} = ${courseId}`)
            .limit(1);
          
          if (courseResult[0]) {
            courseName = courseResult[0].title;
          }
        }

        engagementRanking.push({
          userId,
          name: `${user[0].firstName} ${user[0].lastName}`,
          courseName,
          progress: Math.round(data.totalProgress / data.lessonCount),
          lastActivity: data.lastActivity.toISOString(),
        });
      }
    }

    // Sort by last activity and take top 10
    engagementRanking.sort((a, b) => 
      new Date(b.lastActivity).getTime() - new Date(a.lastActivity).getTime()
    );
    const topEngagement = engagementRanking.slice(0, 10);

    return {
      metrics: {
        totalStudents: totalStudents[0]?.count || 0,
        certificatesIssued: certificatesIssued[0]?.count || 0,
        invalidEmails: invalidEmails[0]?.count || 0,
        accessLast30Days: accessLast30Days[0]?.count || 0,
      },
      studentsByMonth: studentsByMonth.length > 0 ? studentsByMonth : [],
      accessSegmentation,
      pointsRanking,
      engagementRanking: topEngagement,
    };
  }
}

export const storage = new DatabaseStorage();
