// API Routes - Complete implementation for Portal de Cursos
import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { isAuthenticated, isAdmin } from "./authMiddleware";
import { setupAuthRoutes } from "./authRoutes";
import { multer } from "./multer-import";
import { z } from "zod";
import {
  insertCourseSchema,
  insertSaleSchema,
  courseModules,
  userCourseAccess,
  insertCourseShowcaseSchema,
  insertCourseShowcaseCourseSchema,
} from "@shared/schema";
import pkg from "xlsx";
const xlsx = pkg.default || pkg;
import { db } from "./db";
import { eq, and } from "drizzle-orm";
import fs from "fs/promises";
import path from "path";
import crypto from "crypto";

// Configure multer for file uploads (store in memory, then save to database as BYTEA)
const upload = multer({ storage: multer.memoryStorage() });

const miniaturesDir = process.env.PATH_MINIATURAS;
const pdfsDir = process.env.PATH_PDF_CURSOS;

// Helper function to convert thumbnail URL to 550x550-resized format
function convertThumbnailUrlForStudentView(thumbnailUrl: string | null | undefined): string | null {
  if (!thumbnailUrl) return null;
  
  try {
    // Handle URL with query parameters: https://cdn.areademembros.com/image?p=instancia_308%2Fimage%2FaIJIlLmClVEr3hSBFAddjrLCgJArBnuCiHhUYS5V.png&w=80&h=80&t=crop...
    if (thumbnailUrl.includes('cdn.areademembros.com/image?')) {
      const urlObj = new URL(thumbnailUrl);
      const pParam = urlObj.searchParams.get('p');
      
      if (pParam) {
        // Decode the p parameter: instancia_308%2Fimage%2FaIJIlLmClVEr3hSBFAddjrLCgJArBnuCiHhUYS5V.png
        const decodedPath = decodeURIComponent(pParam);
        // Extract filename: aIJIlLmClVEr3hSBFAddjrLCgJArBnuCiHhUYS5V.png
        const filename = decodedPath.split('/').pop();
        
        if (filename) {
          // Remove extension temporarily
          const nameWithoutExt = filename.replace(/\.(png|jpg|jpeg|gif|webp)$/i, '');
          const ext = filename.match(/\.(png|jpg|jpeg|gif|webp)$/i)?.[1] || 'png';
          
          // Build new URL: https://cdn.areademembros.com/cache/{name}-550x550-resized.{ext}
          return `https://cdn.areademembros.com/cache/${nameWithoutExt}-550x550-resized.${ext}`;
        }
      }
    }
    
    // Handle URL with -80x80-cropped pattern: ...-80x80-cropped.png
    if (thumbnailUrl.includes('-80x80-cropped')) {
      return thumbnailUrl.replace(/-80x80-cropped/g, '-550x550-resized');
    }
    
    // Handle URL with -80x80- pattern (without -cropped): ...-80x80-.png
    if (thumbnailUrl.includes('-80x80-')) {
      return thumbnailUrl.replace(/-80x80-/g, '-550x550-');
    }
    
    // If URL doesn't match any pattern, return as is
    return thumbnailUrl;
  } catch (error) {
    // If URL parsing fails, return original
    console.error('Error converting thumbnail URL:', error);
    return thumbnailUrl;
  }
}

async function saveFileToDirectory(
  file: Express.Multer.File,
  baseDir: string,
  prefix: string,
) {
  await fs.mkdir(baseDir, { recursive: true });
  const ext = path.extname(file.originalname) || ".bin";
  const safeName = `${prefix}-${Date.now()}-${crypto.randomUUID()}${ext}`;
  const absolutePath = path.join(baseDir, safeName);
  await fs.writeFile(absolutePath, file.buffer);
  const publicDir = path.basename(baseDir);
  const publicPath = path.posix.join("/", publicDir, safeName);
  return { absolutePath, publicPath, filename: safeName };
}

function resolvePublicPathToAbsolute(baseDir: string | undefined, publicPath?: string | null) {
  if (!baseDir || !publicPath) return null;
  const normalized = publicPath.replace(/^\/+/g, "");
  if (!normalized) return null;
  const parts = normalized.split("/");
  const dirName = path.basename(baseDir);
  const relativeParts = parts[0] === dirName ? parts.slice(1) : parts;
  const absolutePath = path.join(baseDir, ...relativeParts);
  const resolvedBase = path.resolve(baseDir);
  const resolvedAbsolute = path.resolve(absolutePath);
  if (!resolvedAbsolute.startsWith(resolvedBase)) {
    console.warn(`Skipping removal of ${publicPath} - outside base dir`);
    return null;
  }
  return resolvedAbsolute;
}

async function deletePublicFile(baseDir: string | undefined, publicPath?: string | null) {
  const absolute = resolvePublicPathToAbsolute(baseDir, publicPath);
  if (!absolute) return;
  try {
    await fs.unlink(absolute);
  } catch (error: any) {
    if (error?.code !== "ENOENT") {
      console.error(`Error deleting file ${absolute}:`, error);
    }
  }
}

export async function registerRoutes(app: Express): Promise<Server> {
  // Setup authentication routes (login, register, logout)
  setupAuthRoutes(app);


  // ============ AUTH ROUTES ============
  // Auth routes are handled by setupAuthRoutes(app) above

  // ============ ADMIN ROUTES ============
  
  // Dashboard metrics
  app.get("/api/admin/dashboard", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const metrics = await storage.getDashboardMetrics();
      res.json(metrics);
    } catch (error) {
      console.error("Error fetching dashboard metrics:", error);
      res.status(500).json({ message: "Failed to fetch metrics" });
    }
  });

  // Courses CRUD
  app.get("/api/admin/courses", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const courses = await storage.getAllCourses();
      res.json(courses);
    } catch (error) {
      console.error("Error fetching courses:", error);
      res.status(500).json({ message: "Failed to fetch courses" });
    }
  });

  app.post("/api/admin/courses", isAuthenticated, isAdmin, upload.single("thumbnail"), async (req, res) => {
    try {
      const courseData: any = {
        title: req.body.title,
        description: req.body.description || null,
        instructorName: req.body.instructorName || null,
        instructorBio: req.body.instructorBio || null,
        status: req.body.status || "draft",
        thumbnail: req.file ? req.file.buffer.toString('base64') : null,
        thumbnailMimeType: req.file ? req.file.mimetype : null,
        thumbnailUrl: req.body.thumbnailUrl || null,
      };

      // Generate thumbVisaoAluno from thumbnailUrl if provided
      if (courseData.thumbnailUrl) {
        courseData.thumbVisaoAluno = convertThumbnailUrlForStudentView(courseData.thumbnailUrl);
      }

      const validatedData = insertCourseSchema.parse(courseData);
      const course = await storage.createCourse(validatedData);
      res.json(course);
    } catch (error) {
      console.error("Error creating course:", error);
      res.status(500).json({ message: "Failed to create course" });
    }
  });

  app.patch("/api/admin/courses/:id", isAuthenticated, isAdmin, upload.single("thumbnail"), async (req, res) => {
    try {
      // Build update data only with provided fields
      const courseData: any = {};
      
      if (req.body.title !== undefined) courseData.title = req.body.title;
      if (req.body.description !== undefined) courseData.description = req.body.description || null;
      if (req.body.instructorName !== undefined) courseData.instructorName = req.body.instructorName || null;
      if (req.body.instructorBio !== undefined) courseData.instructorBio = req.body.instructorBio || null;
      if (req.body.status !== undefined) courseData.status = req.body.status;
      if (req.body.thumbnailUrl !== undefined) {
        courseData.thumbnailUrl = req.body.thumbnailUrl || null;
        // Generate thumbVisaoAluno from thumbnailUrl if provided
        courseData.thumbVisaoAluno = convertThumbnailUrlForStudentView(req.body.thumbnailUrl);
      }

      // Only update thumbnail if a new file was uploaded
      if (req.file) {
        courseData.thumbnail = req.file.buffer.toString('base64');
        courseData.thumbnailMimeType = req.file.mimetype;
      }

      // Validate with partial schema
      const partialSchema = insertCourseSchema.partial();
      const validatedData = partialSchema.parse(courseData);

      const course = await storage.updateCourse(req.params.id, validatedData);
      res.json(course);
    } catch (error) {
      console.error("Error updating course:", error);
      res.status(500).json({ message: "Failed to update course" });
    }
  });

  app.delete("/api/admin/courses/:id", isAuthenticated, isAdmin, async (req, res) => {
    try {
      await storage.deleteCourse(req.params.id);
      res.json({ message: "Course deleted successfully" });
    } catch (error) {
      console.error("Error deleting course:", error);
      res.status(500).json({ message: "Failed to delete course" });
    }
  });

  // Course showcases (Vitrines)
  app.get("/api/admin/vitrines", isAuthenticated, isAdmin, async (_req, res) => {
    try {
      const showcases = await storage.getAllCourseShowcases();
      res.json(showcases);
    } catch (error) {
      console.error("Error fetching showcases:", error);
      res.status(500).json({ message: "Failed to fetch vitrines" });
    }
  });

  app.post("/api/admin/vitrines", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const showcaseData = insertCourseShowcaseSchema.parse({
        name: req.body.name,
        description: req.body.description ?? null,
        orderIndex: req.body.orderIndex ?? undefined,
        status: req.body.status ?? undefined,
      });

      const showcase = await storage.createCourseShowcase(showcaseData);
      res.json(showcase);
    } catch (error) {
      console.error("Error creating showcase:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid vitrine data", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to create vitrine" });
    }
  });

  app.patch("/api/admin/vitrines/order", isAuthenticated, isAdmin, async (req, res) => {
    const reorderSchema = z.object({
      showcaseIds: z.array(z.string().min(1, "Showcase id is required")).min(1),
    });

    try {
      const { showcaseIds } = reorderSchema.parse(req.body);
      await storage.reorderCourseShowcases(showcaseIds);
      res.json({ success: true });
    } catch (error) {
      console.error("Error reordering showcases:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid reorder payload", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to reorder vitrines" });
    }
  });

  app.patch("/api/admin/vitrines/:id/status", isAuthenticated, isAdmin, async (req, res) => {
    const statusSchema = z.object({ status: z.enum(["listed", "unlisted"]) });

    try {
      const { status } = statusSchema.parse(req.body);
      const showcase = await storage.updateCourseShowcase(req.params.id, { status });
      res.json(showcase);
    } catch (error) {
      console.error("Error updating showcase status:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid status payload", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to update vitrine status" });
    }
  });

  app.patch("/api/admin/vitrines/:id/basic", isAuthenticated, isAdmin, async (req, res) => {
    const schema = z.object({
      name: z.string().min(1, "Nome é obrigatório"),
      description: z.string().optional(),
    });

    try {
      const data = schema.parse(req.body);
      const updated = await storage.updateCourseShowcase(req.params.id, data);
      res.json(updated);
    } catch (error) {
      console.error("Error updating showcase:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid vitrine payload", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to update vitrine" });
    }
  });

  app.delete("/api/admin/vitrines/:id", isAuthenticated, isAdmin, async (req, res) => {
    try {
      await storage.deleteCourseShowcase(req.params.id);
      res.json({ message: "Showcase deleted successfully" });
    } catch (error) {
      console.error("Error deleting showcase:", error);
      res.status(500).json({ message: "Failed to delete vitrine" });
    }
  });

  app.post("/api/admin/vitrines/:id/courses", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const linkData = insertCourseShowcaseCourseSchema.parse({
        showcaseId: req.params.id,
        courseId: req.body.courseId,
        orderIndex: req.body.orderIndex ?? undefined,
      });

      const link = await storage.addCourseToShowcase(linkData);
      res.json(link);
    } catch (error) {
      console.error("Error adding course to showcase:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid vitrine course data", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to attach course to vitrine" });
    }
  });

  app.post("/api/admin/vitrines/:id/courses/simple", isAuthenticated, isAdmin, async (req, res) => {
    const createSimpleCourseSchema = z.object({
      title: z.string().min(1, "Nome do curso é obrigatório"),
      category: z.string().min(1, "Categoria é obrigatória"),
      thumbnailUrl: z.string().url("Informe uma URL válida para o thumbnail"),
      status: z.enum(["draft", "published", "archived"]).optional(),
    });

    try {
      const data = createSimpleCourseSchema.parse(req.body);

      const course = await storage.createCourse({
        title: data.title,
        category: data.category,
        thumbnailUrl: data.thumbnailUrl,
        thumbVisaoAluno: convertThumbnailUrlForStudentView(data.thumbnailUrl),
        thumbnail: null,
        thumbnailMimeType: null,
        description: null,
        instructorName: null,
        instructorBio: null,
        status: data.status ?? "draft",
      });

      const link = await storage.addCourseToShowcase({
        showcaseId: req.params.id,
        courseId: course.id,
        orderIndex: req.body.orderIndex ?? undefined,
      });

      res.json({ course, link });
    } catch (error) {
      console.error("Error creating course for showcase:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid course data", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to create course for vitrine" });
    }
  });

  app.patch("/api/admin/courses/:id/basic", isAuthenticated, isAdmin, async (req, res) => {
    const schema = z.object({
      title: z.string().min(1, "Título é obrigatório").optional(),
      status: z.enum(["draft", "published", "archived"]).optional(),
      category: z.string().min(1).optional(),
      thumbnailUrl: z.string().url().optional(),
    });

    try {
      const data = schema.parse(req.body);
      
      // Generate thumbVisaoAluno from thumbnailUrl if provided
      const updateData: any = { ...data };
      if (data.thumbnailUrl !== undefined) {
        updateData.thumbVisaoAluno = convertThumbnailUrlForStudentView(data.thumbnailUrl);
      }
      const cleaned = Object.fromEntries(Object.entries(updateData).filter(([, value]) => value !== undefined));

      if (Object.keys(cleaned).length === 0) {
        res.status(400).json({ message: "No fields provided" });
        return;
      }

      const updated = await storage.updateCourse(req.params.id, cleaned);
      res.json(updated);
    } catch (error) {
      console.error("Error updating course:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid course payload", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to update course" });
    }
  });

  app.patch("/api/admin/vitrines/:id/courses/order", isAuthenticated, isAdmin, async (req, res) => {
    const reorderSchema = z.object({
      courseIds: z.array(z.string().min(1, "Course id is required")).min(1),
    });

    try {
      const { courseIds } = reorderSchema.parse(req.body);
      await storage.reorderShowcaseCourses(req.params.id, courseIds);
      res.json({ success: true });
    } catch (error) {
      console.error("Error reordering showcase courses:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid reorder payload", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to reorder courses in vitrine" });
    }
  });

  app.delete("/api/admin/vitrines/:id/courses/:courseId", isAuthenticated, isAdmin, async (req, res) => {
    try {
      await storage.removeCourseFromShowcase(req.params.id, req.params.courseId);
      res.json({ success: true });
    } catch (error) {
      console.error("Error removing course from showcase:", error);
      res.status(500).json({ message: "Failed to detach course from vitrine" });
    }
  });

  // Users management
  app.get("/api/admin/users", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const users = await storage.getAllUsers();
      res.json(users);
    } catch (error) {
      console.error("Error fetching users:", error);
      res.status(500).json({ message: "Failed to fetch users" });
    }
  });

  // Export users to XLSX
  app.get("/api/admin/users/export", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const users = await storage.getAllUsers();
      
      const data = users.map(user => ({
        ID: user.id,
        Email: user.email || "",
        "Nome": `${user.firstName || ""} ${user.lastName || ""}`.trim(),
        Role: user.role,
        Pontos: user.points,
        "Data Cadastro": user.createdAt ? new Date(user.createdAt).toLocaleDateString("pt-BR") : "",
      }));

      const worksheet = xlsx.utils.json_to_sheet(data);
      const workbook = xlsx.utils.book_new();
      xlsx.utils.book_append_sheet(workbook, worksheet, "Usuários");
      
      const buffer = xlsx.write(workbook, { type: "buffer", bookType: "xlsx" });
      
      res.setHeader("Content-Disposition", `attachment; filename=usuarios_${new Date().toISOString().split("T")[0]}.xlsx`);
      res.setHeader("Content-Type", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
      res.send(buffer);
    } catch (error) {
      console.error("Error exporting users:", error);
      res.status(500).json({ message: "Failed to export users" });
    }
  });

  // Sales
  app.get("/api/admin/sales", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const sales = await storage.getAllSales();
      res.json(sales);
    } catch (error) {
      console.error("Error fetching sales:", error);
      res.status(500).json({ message: "Failed to fetch sales" });
    }
  });

  // Support tickets
  app.get("/api/admin/support/tickets", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const tickets = await storage.getAllTickets();
      res.json(tickets);
    } catch (error) {
      console.error("Error fetching tickets:", error);
      res.status(500).json({ message: "Failed to fetch tickets" });
    }
  });

  // Student support tickets - Get user's own tickets
  app.get("/api/support/tickets", isAuthenticated, async (req, res) => {
    try {
      const userId = req.user?.id;
      if (!userId) {
        res.status(401).json({ message: "Unauthorized" });
        return;
      }
      const allTickets = await storage.getAllTickets();
      const userTickets = allTickets.filter(ticket => ticket.userId === userId);
      res.json(userTickets);
    } catch (error) {
      console.error("Error fetching user tickets:", error);
      res.status(500).json({ message: "Failed to fetch tickets" });
    }
  });

  // Student support tickets - Create new ticket
  app.post("/api/support/tickets", isAuthenticated, async (req, res) => {
    try {
      const userId = req.user?.id;
      const userEmail = req.user?.email;
      
      if (!userId || !userEmail) {
        res.status(401).json({ message: "Unauthorized" });
        return;
      }

      const schema = z.object({
        subject: z.string().min(1, "Assunto é obrigatório"),
        message: z.string().min(1, "Mensagem é obrigatória"),
        priority: z.enum(["low", "medium", "high", "urgent"]).default("medium"),
      });

      const data = schema.parse(req.body);
      
      const ticket = await storage.createTicket({
        userId,
        userEmail,
        subject: data.subject,
        message: data.message,
        priority: data.priority,
        status: "open",
      });

      res.json(ticket);
    } catch (error) {
      console.error("Error creating ticket:", error);
      if (error instanceof z.ZodError) {
        res.status(400).json({ message: "Invalid ticket data", issues: error.issues });
        return;
      }
      res.status(500).json({ message: "Failed to create ticket" });
    }
  });

  // FAQ
  app.get("/api/admin/faq", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const faqs = await storage.getAllFaqs();
      res.json(faqs);
    } catch (error) {
      console.error("Error fetching FAQs:", error);
      res.status(500).json({ message: "Failed to fetch FAQs" });
    }
  });

  // Public FAQ endpoint for students
  app.get("/api/faq", async (req, res) => {
    try {
      const faqs = await storage.getAllFaqs();
      res.json(faqs);
    } catch (error) {
      console.error("Error fetching FAQs:", error);
      res.status(500).json({ message: "Failed to fetch FAQs" });
    }
  });

  // ============ COURSE CONTENT MANAGEMENT ============

  // Get modules for a course
  app.get("/api/admin/courses/:courseId/modules", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const modules = await storage.getModulesByCourse(req.params.courseId);
      res.json(modules);
    } catch (error) {
      console.error("Error fetching modules:", error);
      res.status(500).json({ message: "Failed to fetch modules" });
    }
  });

  // Create module
  app.post("/api/admin/courses/:courseId/modules", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const moduleData = {
        courseId: req.params.courseId,
        title: req.body.title,
        description: req.body.description || null,
        thumbnailUrl: req.body.thumbnailUrl || null,
        orderIndex: req.body.orderIndex || 0,
      };

      const module = await storage.createModule(moduleData);
      res.json(module);
    } catch (error) {
      console.error("Error creating module:", error);
      res.status(500).json({ message: "Failed to create module" });
    }
  });

  // Get lessons for a module
  app.get("/api/admin/modules/:moduleId/lessons", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const lessons = await storage.getLessonsByModule(req.params.moduleId);
      // Don't send video data in list view
      const lessonsWithoutVideo = lessons.map(l => ({ ...l, videoData: undefined }));
      res.json(lessonsWithoutVideo);
    } catch (error) {
      console.error("Error fetching lessons:", error);
      res.status(500).json({ message: "Failed to fetch lessons" });
    }
  });

  app.post(
    "/api/admin/modules/:moduleId/lessons",
    isAuthenticated,
    isAdmin,
    upload.fields([
      { name: "thumbnail", maxCount: 1 },
    ]),
    async (req, res) => {
    try {
      const moduleRecord = await storage.getModule(req.params.moduleId);
      if (!moduleRecord) {
        return res.status(404).json({ message: "Module not found" });
      }

      const thumbnailFiles = (req.files as Record<string, Express.Multer.File[]>)?.thumbnail;
      let storedThumbnailPath: string | null = moduleRecord.thumbnailUrl;

      if (thumbnailFiles && thumbnailFiles[0]) {
        if (!miniaturesDir) {
          return res.status(500).json({ message: "PATH_MINIATURAS não configurado" });
        }
        try {
          const { publicPath } = await saveFileToDirectory(
            thumbnailFiles[0],
            miniaturesDir,
            "lesson-thumb",
          );
          storedThumbnailPath = publicPath;
        } catch (fileError) {
          console.error("Error saving lesson thumbnail:", fileError);
          return res.status(500).json({ message: "Failed to save lesson thumbnail" });
        }
      } else if (req.body.thumbnailUrl) {
        storedThumbnailPath = req.body.thumbnailUrl;
      }

      const lessonData = {
        moduleId: req.params.moduleId,
        title: req.body.title,
        description: req.body.description || null,
        thumbnailUrl: storedThumbnailPath || null,
        videoUrl: req.body.videoUrl || null,
        status: req.body.status || "published",
        orderIndex: req.body.orderIndex || 0,
      };

      const lesson = await storage.createLesson(lessonData);
      res.json(lesson);
    } catch (error) {
      console.error("Error creating lesson:", error);
      res.status(500).json({ message: "Failed to create lesson" });
    }
  });

  app.patch("/api/admin/lessons/:lessonId/basic", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const updates: Record<string, any> = {};
      if (req.body.title !== undefined) updates.title = req.body.title;
      if (req.body.description !== undefined) updates.description = req.body.description ?? null;
      if (req.body.thumbnailUrl !== undefined) updates.thumbnailUrl = req.body.thumbnailUrl || null;
      if (req.body.videoUrl !== undefined) updates.videoUrl = req.body.videoUrl || null;

      if (Object.keys(updates).length === 0) {
        return res.status(400).json({ message: "No fields provided" });
      }

      const lesson = await storage.updateLesson(req.params.lessonId, updates);
      res.json(lesson);
    } catch (error) {
      console.error("Error updating lesson:", error);
      res.status(500).json({ message: "Failed to update lesson" });
    }
  });

  app.patch("/api/admin/lessons/:lessonId/status", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const { status } = req.body;
      if (!status || !["published", "unlisted"].includes(status)) {
        return res.status(400).json({ message: "Invalid status" });
      }

      const lesson = await storage.updateLessonStatus(req.params.lessonId, status);
      res.json(lesson);
    } catch (error) {
      console.error("Error updating lesson status:", error);
      res.status(500).json({ message: "Failed to update lesson status" });
    }
  });

  app.get("/api/admin/lessons/:lessonId/materials", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const materials = await storage.getMaterialsByLesson(req.params.lessonId);
      const sanitized = materials.map(material => ({
        id: material.id,
        title: material.title,
        description: material.description,
        fileName: material.fileName,
        fileMimeType: material.fileMimeType,
        fileSize: material.fileSize,
        materialType: material.materialType,
        filePath: material.filePath,
        createdAt: material.createdAt,
      }));
      res.json(sanitized);
    } catch (error) {
      console.error("Error fetching lesson materials:", error);
      res.status(500).json({ message: "Failed to fetch lesson materials" });
    }
  });

  app.post("/api/admin/lessons/:lessonId/materials", isAuthenticated, isAdmin, upload.single("file"), async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ message: "File is required" });
      }

      const lesson = await storage.getLesson(req.params.lessonId);
      if (!lesson) {
        return res.status(404).json({ message: "Lesson not found" });
      }

      const moduleRecord = await storage.getModule(lesson.moduleId);
      if (!moduleRecord) {
        return res.status(404).json({ message: "Module not found" });
      }

      if (!pdfsDir) {
        return res.status(500).json({ message: "PATH_PDF_CURSOS não configurado" });
      }

      let publicPath: string;
      try {
        const saved = await saveFileToDirectory(req.file, pdfsDir, "material");
        publicPath = saved.publicPath;
      } catch (fileError) {
        console.error("Error saving lesson material:", fileError);
        return res.status(500).json({ message: "Failed to save material file" });
      }

      const materialData = {
        courseId: moduleRecord.courseId,
        lessonId: lesson.id,
        title: req.body.title,
        description: req.body.description || null,
        filePath: publicPath,
        fileName: req.file.originalname,
        fileMimeType: req.file.mimetype,
        fileSize: req.file.size,
        materialType: req.body.materialType || "pdf",
      };

      const material = await storage.createMaterial(materialData);
      res.json(material);
    } catch (error) {
      console.error("Error creating lesson material:", error);
      res.status(500).json({ message: "Failed to create lesson material" });
    }
  });

  app.delete("/api/admin/lessons/:lessonId/materials/:materialId", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const material = await storage.getMaterial(req.params.materialId);
      if (!material || material.lessonId !== req.params.lessonId) {
        return res.status(404).json({ message: "Material not found" });
      }

      await deletePublicFile(pdfsDir, material.filePath);
      await storage.deleteMaterial(req.params.materialId);
      res.status(204).send();
    } catch (error) {
      console.error("Error deleting lesson material:", error);
      res.status(500).json({ message: "Failed to delete lesson material" });
    }
  });

  app.delete("/api/admin/lessons/:lessonId", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const lesson = await storage.getLesson(req.params.lessonId);
      if (!lesson) {
        return res.status(404).json({ message: "Lesson not found" });
      }

      const materials = await storage.getMaterialsByLesson(req.params.lessonId);
      for (const material of materials) {
        await deletePublicFile(pdfsDir, material.filePath);
        await storage.deleteMaterial(material.id);
      }

      await deletePublicFile(miniaturesDir, lesson.thumbnailUrl);
      await storage.deleteLesson(req.params.lessonId);

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting lesson:", error);
      res.status(500).json({ message: "Failed to delete lesson" });
    }
  });

  app.delete("/api/admin/modules/:moduleId", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const moduleRecord = await storage.getModule(req.params.moduleId);
      if (!moduleRecord) {
        return res.status(404).json({ message: "Module not found" });
      }

      const lessons = await storage.getLessonsByModule(req.params.moduleId);
      for (const lesson of lessons) {
        const materials = await storage.getMaterialsByLesson(lesson.id);
        for (const material of materials) {
          await deletePublicFile(pdfsDir, material.filePath);
          await storage.deleteMaterial(material.id);
        }
        await deletePublicFile(miniaturesDir, lesson.thumbnailUrl);
        await storage.deleteLesson(lesson.id);
      }

      await deletePublicFile(miniaturesDir, moduleRecord.thumbnailUrl);
      await storage.deleteModule(req.params.moduleId);

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting module:", error);
      res.status(500).json({ message: "Failed to delete module" });
    }
  });

  // Get user's course access
  app.get("/api/admin/users/:userId/courses", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const access = await storage.getUserCourseAccess(req.params.userId);
      const courseIds = access.map(a => a.courseId);
      res.json(courseIds);
    } catch (error) {
      console.error("Error fetching user course access:", error);
      res.status(500).json({ message: "Failed to fetch user course access" });
    }
  });

  // Grant course access to user
  app.post("/api/admin/user-access", isAuthenticated, isAdmin, async (req, res) => {
    try {
      const accessData = {
        userId: req.body.userId,
        courseId: req.body.courseId,
      };

      const access = await storage.grantCourseAccess(accessData);
      res.json(access);
    } catch (error) {
      console.error("Error granting access:", error);
      res.status(500).json({ message: "Failed to grant access" });
    }
  });

  // Revoke course access from user
  app.delete("/api/admin/user-access/:userId/:courseId", isAuthenticated, isAdmin, async (req, res) => {
    try {
      await db
        .delete(userCourseAccess)
        .where(and(
          eq(userCourseAccess.userId, req.params.userId),
          eq(userCourseAccess.courseId, req.params.courseId)
        ));
      res.json({ success: true });
    } catch (error) {
      console.error("Error revoking access:", error);
      res.status(500).json({ message: "Failed to revoke access" });
    }
  });

  // ============ STUDENT ROUTES ============

  // Get student courses (courses they have access to + available courses)
  app.get("/api/student/courses", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user.id;
      
      // Get courses user has access to
      const userAccess = await storage.getUserCourseAccess(userId);
      const accessedCourseIds = userAccess.map(a => a.courseId);
      
      const allCourses = await storage.getAllCourses();
      const publishedCourses = allCourses.filter(c => c.status === "published");
      
      // Get my courses with progress
      const myCourses = [];
      for (const courseId of accessedCourseIds) {
        const course = publishedCourses.find(c => c.id === courseId);
        if (course) {
          const progress = await storage.getCourseProgress(userId, courseId);
          
          // Calculate overall progress
          const modules = await storage.getModulesByCourse(courseId);
          let totalLessons = 0;
          for (const module of modules) {
            const lessons = await storage.getLessonsByModule(module.id);
            totalLessons += lessons.length;
          }
          
          const completedLessons = progress.filter(p => p.completed).length;
          const progressPercentage = totalLessons > 0 ? Math.round((completedLessons / totalLessons) * 100) : 0;
          
          myCourses.push({ ...course, progressPercentage });
        }
      }
      
      // Get available courses (not yet accessed)
      const availableCourses = publishedCourses.filter(c => !accessedCourseIds.includes(c.id));
      
      // Featured course (first in-progress course or first available)
      const featuredCourse = myCourses.find(c => c.progressPercentage > 0 && c.progressPercentage < 100) || myCourses[0] || null;
      
      res.json({
        myCourses,
        availableCourses,
        featuredCourse,
      });
    } catch (error) {
      console.error("Error fetching student courses:", error);
      res.status(500).json({ message: "Failed to fetch courses" });
    }
  });

  // Get student vitrines (showcases) with courses
  app.get("/api/student/vitrines", isAuthenticated, async (_req: any, res) => {
    try {
      const allShowcases = await storage.getAllCourseShowcases();
      // Filter only listed showcases
      const listedShowcases = allShowcases.filter(s => s.status === "listed");
      
      // Filter only published courses in each showcase
      const showcasesWithPublishedCourses = listedShowcases.map(showcase => ({
        id: showcase.id,
        name: showcase.name,
        description: showcase.description,
        orderIndex: showcase.orderIndex,
        courses: showcase.courses.filter(c => c.status === "published"),
      }));
      
      // Sort by orderIndex
      showcasesWithPublishedCourses.sort((a, b) => a.orderIndex - b.orderIndex);
      
      res.json(showcasesWithPublishedCourses);
    } catch (error) {
      console.error("Error fetching student vitrines:", error);
      res.status(500).json({ message: "Failed to fetch vitrines" });
    }
  });

  // Get course details with modules, lessons, and progress
  app.get("/api/student/course/:id", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user.id;
      const courseId = req.params.id;
      
      const course = await storage.getCourse(courseId);
      if (!course) {
        return res.status(404).json({ message: "Course not found" });
      }
      
      const modules = await storage.getModulesByCourse(courseId);
      const modulesWithLessons = [];
      let totalLessons = 0;
      let completedLessons = 0;
      
      for (const module of modules) {
        const lessons = await storage.getLessonsByModule(module.id);
        totalLessons += lessons.length;
        
        const lessonsWithProgress = [];
        for (const lesson of lessons) {
          const progress = await storage.getStudentProgress(userId, lesson.id);
          if (progress?.completed) completedLessons++;
          
          lessonsWithProgress.push({
            ...lesson,
            videoData: undefined, // Don't send video data in list
            videoUrl: lesson.videoUrl || null, // Include videoUrl to check if lesson has video
            completed: progress?.completed || false,
            progressPercentage: progress?.progressPercentage || 0,
          });
        }
        
        modulesWithLessons.push({
          ...module,
          lessons: lessonsWithProgress,
        });
      }
      
      const overallProgress = totalLessons > 0 ? Math.round((completedLessons / totalLessons) * 100) : 0;
      
      const materials = await storage.getMaterialsByCourse(courseId);
      const materialsWithoutData = materials.map(m => ({
        id: m.id,
        title: m.title,
        fileName: m.fileName,
        fileMimeType: m.fileMimeType,
        fileSize: m.fileSize,
      }));
      
      res.json({
        course: {
          id: course.id,
          title: course.title,
          description: course.description,
          instructorName: course.instructorName,
          instructorBio: course.instructorBio,
        },
        modules: modulesWithLessons,
        materials: materialsWithoutData,
        overallProgress,
      });
    } catch (error) {
      console.error("Error fetching course details:", error);
      res.status(500).json({ message: "Failed to fetch course details" });
    }
  });

  // Get lesson video (with access control)
  app.get("/api/student/lesson/:lessonId/video", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user.id;
      const lesson = await storage.getLesson(req.params.lessonId);
      
      if (!lesson) {
        return res.status(404).json({ message: "Lesson not found" });
      }
      
      // Get module and course via direct queries (efficient)
      const modules = await db.select().from(courseModules).where(eq(courseModules.id, lesson.moduleId));
      const module = modules[0];
      
      if (!module) {
        return res.status(404).json({ message: "Course not found" });
      }
      
      // Check if user has access to this course
      const access = await db
        .select()
        .from(userCourseAccess)
        .where(and(
          eq(userCourseAccess.userId, userId),
          eq(userCourseAccess.courseId, module.courseId)
        ));
      
      if (access.length === 0) {
        return res.status(403).json({ message: "Access denied" });
      }
      
      // If video is stored as base64 (videoData)
      if (lesson.videoData) {
        res.setHeader("Content-Type", lesson.videoMimeType || "video/mp4");
        res.send(Buffer.from(lesson.videoData, 'base64'));
        return;
      }
      
      // If video is external URL (videoUrl), redirect to it
      if (lesson.videoUrl) {
        res.redirect(lesson.videoUrl);
        return;
      }
      
      // No video found
      return res.status(404).json({ message: "Video not found" });
    } catch (error) {
      console.error("Error fetching video:", error);
      res.status(500).json({ message: "Failed to fetch video" });
    }
  });

  // Download course material
  app.get("/api/student/course/:courseId/material/:materialId", isAuthenticated, async (req, res) => {
    try {
      const material = await storage.getMaterial(req.params.materialId);
      
      if (!material) {
        return res.status(404).json({ message: "Material not found" });
      }
      
      res.setHeader("Content-Disposition", `attachment; filename="${material.fileName}"`);
      res.setHeader("Content-Type", material.fileMimeType);
      res.send(Buffer.from(material.fileData, 'base64'));
    } catch (error) {
      console.error("Error downloading material:", error);
      res.status(500).json({ message: "Failed to download material" });
    }
  });

  // Update student progress
  app.post("/api/student/progress", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user.id;
      const { lessonId, completed, progressPercentage } = req.body;
      
      const progress = await storage.upsertStudentProgress({
        userId,
        lessonId,
        completed: completed || false,
        progressPercentage: progressPercentage || 0,
      });
      
      res.json(progress);
    } catch (error) {
      console.error("Error updating progress:", error);
      res.status(500).json({ message: "Failed to update progress" });
    }
  });

  // ============ WEBHOOK ROUTES ============

  // Webhook for sales from external platform
  app.post("/api/webhook/sales", async (req, res) => {
    try {
      const saleData = {
        webhookId: req.body.webhookId || req.body.id,
        courseId: req.body.courseId || null,
        buyerEmail: req.body.buyerEmail || req.body.email,
        status: "pending" as "pending" | "processed" | "error",
        amount: req.body.amount ? req.body.amount.toString() : null,
        currency: req.body.currency || "BRL",
        purchasedAt: req.body.purchasedAt ? new Date(req.body.purchasedAt) : new Date(),
      };

      const sale = await storage.createSale(saleData);
      res.json({ success: true, saleId: sale.id });
    } catch (error) {
      console.error("Error processing webhook:", error);
      res.status(500).json({ message: "Failed to process webhook" });
    }
  });

  const httpServer = createServer(app);
  return httpServer;
}
