import "dotenv/config";
import express, { type Request, Response, NextFunction } from "express";
import { session } from "./session-import";
import passport from "./auth";
import { registerRoutes } from "./routes";
import { setupVite, serveStatic, log } from "./vite";
import { connectPgSimple } from "./connect-pg-import";
import pkg2 from "pg";
const { Pool } = pkg2;

const app = express();
app.use(express.json());
app.use(express.urlencoded({ extended: false }));

const miniaturesDir = process.env.PATH_MINIATURAS;
const pdfsDir = process.env.PATH_PDF_CURSOS;

if (!miniaturesDir) {
  console.warn("PATH_MINIATURAS não configurado. Miniaturas não serão servidas estaticamente.");
}

if (!pdfsDir) {
  console.warn("PATH_PDF_CURSOS não configurado. PDFs não serão servidos estaticamente.");
}

if (miniaturesDir) {
  app.use("/miniaturas", express.static(miniaturesDir));
}

if (pdfsDir) {
  app.use("/pdfs", express.static(pdfsDir));
}

// Session configuration for VPS deployment
const PgSession = connectPgSimple(session);
const pool = new Pool({ connectionString: process.env.DATABASE_URL });

app.use(
  session({
    store: new PgSession({
      pool: pool as any,
      tableName: "sessions",
      createTableIfMissing: false,
    }),
    secret: process.env.SESSION_SECRET || "your-secret-key-change-in-production",
    resave: false,
    saveUninitialized: false,
    cookie: {
      maxAge: 30 * 24 * 60 * 60 * 1000, // 30 days
      httpOnly: true,
      secure: process.env.USE_HTTPS === "true",
      sameSite: "lax",
    },
  })
);

// Initialize Passport
app.use(passport.initialize());
app.use(passport.session());

// Logout route - must be before other routes
app.get("/logout", (req, res) => {
  console.log("Rota /logout acessada");
  // Destroy session
  req.session.destroy((err) => {
    if (err) {
      console.error("Erro ao destruir sessão:", err);
      return res.redirect("/?error=logout_failed");
    }
    
    // Clear session cookie
    res.clearCookie('connect.sid');
    
    console.log("Logout realizado com sucesso");
    res.redirect("/?message=logout_success");
  });
});

// Alternative logout route via API
app.get("/api/logout-redirect", (req, res) => {
  console.log("Rota /api/logout-redirect acessada");
  // Destroy session
  req.session.destroy((err) => {
    if (err) {
      console.error("Erro ao destruir sessão:", err);
      return res.redirect("/?error=logout_failed");
    }
    
    // Clear session cookie
    res.clearCookie('connect.sid');
    
    console.log("Logout realizado com sucesso via API");
    res.redirect("/?message=logout_success");
  });
});

app.use((req, res, next) => {
  const start = Date.now();
  const path = req.path;
  let capturedJsonResponse: Record<string, any> | undefined = undefined;

  const originalResJson = res.json;
  res.json = function (bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };

  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path.startsWith("/api")) {
      let logLine = `${req.method} ${path} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }

      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "…";
      }

      log(logLine);
    }
  });

  next();
});

(async () => {
  const server = await registerRoutes(app);

  app.use((err: any, _req: Request, res: Response, _next: NextFunction) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";

    res.status(status).json({ message });
    throw err;
  });

  // importantly only setup vite in development and after
  // setting up all the other routes so the catch-all route
  // doesn't interfere with the other routes
  if (app.get("env") === "development") {
    await setupVite(app, server);
  } else {
    serveStatic(app);
  }

  // ALWAYS serve the app on the port specified in the environment variable PORT
  // Other ports are firewalled. Default to 5000 if not specified.
  // this serves both the API and the client.
  // It is the only port that is not firewalled.
  const port = parseInt(process.env.PORT || '5000', 10);
  server.listen({
    port,
    host: "0.0.0.0",
    reusePort: true,
  }, () => {
    log(`serving on port ${port}`);
  });
})();
