// Authentication Routes for Email/Password login (VPS deployment)
import type { Express } from "express";
import passport from "./auth";
import { hashPassword } from "./auth";
import { storage } from "./storage";
import { z } from "zod";
import { isAuthenticated } from "./authMiddleware";

// Validation schemas
const registerSchema = z.object({
  email: z.string().email("Email inválido"),
  password: z.string().min(6, "Senha deve ter no mínimo 6 caracteres"),
  firstName: z.string().optional(),
  lastName: z.string().optional(),
  role: z.enum(["admin", "student"]).optional(),
});

const loginSchema = z.object({
  email: z.string().email("Email inválido"),
  password: z.string().min(1, "Senha é obrigatória"),
});

export function setupAuthRoutes(app: Express) {
  // Register new user
  app.post("/api/register", async (req, res) => {
    try {
      const data = registerSchema.parse(req.body);
      
      // Check if user already exists
      const existingUsers = await storage.getAllUsers();
      const userExists = existingUsers.find(u => u.email === data.email);
      
      if (userExists) {
        return res.status(400).json({ message: "Email já cadastrado" });
      }

      // Hash password
      const hashedPassword = await hashPassword(data.password);

      // Create user
      const user = await storage.upsertUser({
        email: data.email,
        password: hashedPassword,
        firstName: data.firstName || null,
        lastName: data.lastName || null,
        role: data.role || "student",
      });

      // Convert to Express.User format
      const userForSession = {
        ...user,
        firstName: user.firstName || undefined,
        lastName: user.lastName || undefined,
      };

      // Auto-login after registration
      req.login(userForSession as any, (err) => {
        if (err) {
          console.error("Erro no auto-login:", err);
          return res.status(500).json({ message: "Erro ao fazer login automático" });
        }
        
        console.log("Auto-login realizado com sucesso para:", user.email);
        
        // Don't send password back
        const { password, ...userWithoutPassword } = user;
        res.json(userWithoutPassword);
      });
    } catch (error: any) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: error.errors[0].message });
      }
      console.error("Erro no registro:", error);
      res.status(500).json({ message: "Erro ao registrar usuário" });
    }
  });

  // Login
  app.post("/api/login", (req, res, next) => {
    try {
      const data = loginSchema.parse(req.body);
      
      passport.authenticate("local", (err: any, user: any, info: any) => {
        if (err) {
          console.error("Erro na autenticação:", err);
          return res.status(500).json({ message: "Erro no servidor" });
        }
        
        if (!user) {
          return res.status(401).json({ message: info?.message || "Email ou senha inválidos" });
        }

        req.login(user, (err) => {
          if (err) {
            console.error("Erro ao criar sessão:", err);
            return res.status(500).json({ message: "Erro ao fazer login" });
          }
          
          // Don't send password back
          const { password, ...userWithoutPassword } = user;
          res.json(userWithoutPassword);
        });
      })(req, res, next);
    } catch (error: any) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: error.errors[0].message });
      }
      res.status(500).json({ message: "Erro ao fazer login" });
    }
  });

  // Logout
  app.post("/api/logout", (req, res) => {
    req.logout((err) => {
      if (err) {
        return res.status(500).json({ message: "Erro ao fazer logout" });
      }
      res.json({ message: "Logout realizado com sucesso" });
    });
  });


  // Get current user
  app.get("/api/auth/user", isAuthenticated, async (req, res) => {
    try {
      if (!req.user) {
        return res.status(401).json({ message: "Não autenticado" });
      }

      // Log access for dashboard metrics
      await storage.logAccess(req.user.id);
      
      // Don't send password back
      const user: any = req.user;
      const { password, ...userWithoutPassword } = user;
      res.json(userWithoutPassword);
    } catch (error) {
      console.error("Erro ao buscar usuário:", error);
      res.status(500).json({ message: "Erro ao buscar usuário" });
    }
  });
}
