// Traditional Email/Password Authentication for VPS deployment
import { Strategy as LocalStrategy } from "passport-local";
import pkg from "bcrypt";
const bcrypt = pkg;
import { db } from "./db";
import { users, type User } from "@shared/schema";
import { eq } from "drizzle-orm";
import { passport } from "./passport-import";

const SALT_ROUNDS = 10;

// Configure passport local strategy
passport.use(
  new LocalStrategy(
    {
      usernameField: "email",
      passwordField: "password",
    },
    async (email, password, done) => {
      try {
        const user = await db.select().from(users).where(eq(users.email, email)).limit(1);
        
        if (!user || user.length === 0) {
          return done(null, false, { message: "Email ou senha inválidos" });
        }

        const foundUser = user[0];
        
        if (!foundUser.password) {
          return done(null, false, { message: "Usuário sem senha configurada. Entre em contato com o administrador." });
        }

        const isValid = await bcrypt.compare(password, foundUser.password);
        
        if (!isValid) {
          return done(null, false, { message: "Email ou senha inválidos" });
        }

        // Convert null to undefined for Express.User type
        const userForSession = {
          ...foundUser,
          firstName: foundUser.firstName || undefined,
          lastName: foundUser.lastName || undefined,
        };

        return done(null, userForSession as any);
      } catch (error) {
        return done(error);
      }
    }
  )
);

// Serialize user into session
passport.serializeUser((user: any, done) => {
  done(null, user.id);
});

// Deserialize user from session
passport.deserializeUser(async (id: string, done) => {
  try {
    const user = await db.select().from(users).where(eq(users.id, id)).limit(1);
    if (!user || user.length === 0) {
      return done(null, false);
    }
    // Convert null to undefined for Express.User type
    const userForSession = {
      ...user[0],
      firstName: user[0].firstName || undefined,
      lastName: user[0].lastName || undefined,
    };
    done(null, userForSession as any);
  } catch (error) {
    done(error);
  }
});

// Helper function to hash passwords
export async function hashPassword(password: string): Promise<string> {
  return bcrypt.hash(password, SALT_ROUNDS);
}

// Helper function to validate passwords
export async function validatePassword(password: string, hash: string): Promise<boolean> {
  return bcrypt.compare(password, hash);
}

export default passport;
