// Seed script to populate initial data
import { config } from "dotenv";
import { resolve } from "path";

// Load environment variables from .env file
config({ path: resolve(process.cwd(), '.env') });

import { db } from "./server/db";
import { faqItems, users, sales } from "@shared/schema";
import { hashPassword } from "./server/auth";
import { eq } from "drizzle-orm";

async function seed() {
  console.log("🌱 Seeding database...");
  console.log("DATABASE_URL:", process.env.DATABASE_URL ? "✅ Set" : "❌ Not set");

  // Create admin user if it doesn't exist
  const adminEmail = process.env.ADMIN_EMAIL || "admin@saberon.com";
  const adminPassword = process.env.ADMIN_PASSWORD || "admin123";
  const adminFirstName = process.env.ADMIN_FIRST_NAME || "Admin";
  const adminLastName = process.env.ADMIN_LAST_NAME || "Sistema";

  console.log("🔐 Checking for admin user...");
  
  try {
    const existingAdmin = await db.select().from(users).where(eq(users.email, adminEmail)).limit(1);
    
    if (existingAdmin.length === 0) {
      console.log("👤 Creating admin user...");
      const hashedPassword = await hashPassword(adminPassword);
      
      await db.insert(users).values({
        email: adminEmail,
        password: hashedPassword,
        firstName: adminFirstName,
        lastName: adminLastName,
        role: "admin",
        points: 0,
      });
      
      console.log("✅ Admin user created successfully!");
      console.log(`📧 Email: ${adminEmail}`);
      console.log(`🔑 Password: ${adminPassword}`);
      console.log("⚠️  Please change the admin password after first login!");
    } else {
      console.log("✅ Admin user already exists");
    }
  } catch (error) {
    console.error("❌ Error creating admin user:", error);
  }

  // Add FAQ items
  const faqs = [
    {
      question: "Como faço para acessar meus cursos?",
      answer: "Após fazer login, você será direcionado automaticamente para o dashboard onde poderá ver todos os cursos disponíveis. Clique em qualquer curso para começar a assistir.",
      category: "Acesso",
      orderIndex: 1,
      isPublished: true,
    },
    {
      question: "Como funciona o sistema de certificados?",
      answer: "Ao completar 100% das aulas de um curso, você receberá automaticamente um certificado digital que pode ser baixado e compartilhado.",
      category: "Certificados",
      orderIndex: 2,
      isPublished: true,
    },
    {
      question: "Posso assistir as aulas offline?",
      answer: "Atualmente, as aulas precisam ser assistidas online. Estamos trabalhando para adicionar a funcionalidade de download em breve.",
      category: "Funcionalidades",
      orderIndex: 3,
      isPublished: true,
    },
    {
      question: "Como acompanho meu progresso?",
      answer: "Seu progresso é salvo automaticamente. Você pode ver o percentual de conclusão de cada curso no dashboard e na página do curso.",
      category: "Progresso",
      orderIndex: 4,
      isPublished: true,
    },
    {
      question: "Preciso pagar para acessar os cursos?",
      answer: "Alguns cursos são gratuitos, outros requerem compra. Entre em contato com o administrador para obter acesso aos cursos pagos.",
      category: "Pagamentos",
      orderIndex: 5,
      isPublished: true,
    },
  ];

  // Clear existing FAQs first to avoid duplicates
  await db.delete(faqItems);
  
  // Insert FAQ items
  for (const faq of faqs) {
    await db.insert(faqItems).values(faq);
  }

  // Add sample sales data
  console.log("💰 Adding sample sales data...");
  
  const sampleSales = [
    {
      buyerEmail: "maria.silva@email.com",
      courseId: null, // Some sales might not have course ID
      status: "processed" as const,
      amount: "299.90",
      currency: "BRL",
      purchasedAt: new Date("2024-01-15T10:30:00Z"),
      processedAt: new Date("2024-01-15T10:35:00Z"),
      webhookId: "wh_1234567890abcdef",
    },
    {
      buyerEmail: "joao.santos@empresa.com.br",
      courseId: null,
      status: "processed" as const,
      amount: "199.90",
      currency: "BRL",
      purchasedAt: new Date("2024-01-20T14:15:00Z"),
      processedAt: new Date("2024-01-20T14:20:00Z"),
      webhookId: "wh_abcdef1234567890",
    },
    {
      buyerEmail: "ana.costa@universidade.edu",
      courseId: null,
      status: "pending" as const,
      amount: "399.90",
      currency: "BRL",
      purchasedAt: new Date("2024-01-25T09:45:00Z"),
      processedAt: null,
      webhookId: "wh_pending123456789",
    },
    {
      buyerEmail: "carlos.oliveira@startup.com",
      courseId: null,
      status: "processed" as const,
      amount: "149.90",
      currency: "BRL",
      purchasedAt: new Date("2024-01-28T16:20:00Z"),
      processedAt: new Date("2024-01-28T16:25:00Z"),
      webhookId: "wh_processed987654321",
    },
    {
      buyerEmail: "lucia.ferreira@consultoria.com",
      courseId: null,
      status: "error" as const,
      amount: "599.90",
      currency: "BRL",
      purchasedAt: new Date("2024-01-30T11:10:00Z"),
      processedAt: null,
      webhookId: "wh_error9876543210",
    },
    {
      buyerEmail: "pedro.mendes@freelancer.com",
      courseId: null,
      status: "processed" as const,
      amount: "249.90",
      currency: "BRL",
      purchasedAt: new Date("2024-02-01T13:30:00Z"),
      processedAt: new Date("2024-02-01T13:35:00Z"),
      webhookId: "wh_february123456789",
    },
    {
      buyerEmail: "fernanda.lima@empresa.com.br",
      courseId: null,
      status: "processed" as const,
      amount: "179.90",
      currency: "BRL",
      purchasedAt: new Date("2024-02-03T15:45:00Z"),
      processedAt: new Date("2024-02-03T15:50:00Z"),
      webhookId: "wh_february987654321",
    },
    {
      buyerEmail: "rafael.souza@startup.com",
      courseId: null,
      status: "pending" as const,
      amount: "329.90",
      currency: "BRL",
      purchasedAt: new Date("2024-02-05T08:20:00Z"),
      processedAt: null,
      webhookId: "wh_pending987654321",
    },
  ];

  // Clear existing sales first
  await db.delete(sales);
  
  // Insert sample sales
  for (const sale of sampleSales) {
    await db.insert(sales).values(sale);
  }

  console.log("✅ Database seeded successfully!");
  process.exit(0);
}

seed().catch((error) => {
  console.error("❌ Seeding failed:", error);
  process.exit(1);
});
