#!/usr/bin/env tsx

/**
 * Inicializa o banco de dados aplicando o schema e executando o seed.
 * - Carrega as variáveis do arquivo .env (caso exista)
 * - Garante que DATABASE_URL esteja definido
 * - Executa `drizzle-kit push` para criar/atualizar tabelas e colunas
 * - Roda `tsx seed.ts` para popular dados iniciais
 */

import { resolve } from "path";
import { existsSync } from "fs";
import { execSync } from "node:child_process";
import { config } from "dotenv";

const envPath = resolve(process.cwd(), ".env");
if (existsSync(envPath)) {
  config({ path: envPath });
  console.log(`🔐 Variáveis carregadas de ${envPath}`);
} else {
  console.log("⚠️  Arquivo .env não encontrado. Prosseguindo com as variáveis atuais.");
}

if (!process.env.DATABASE_URL) {
  console.error("❌ DATABASE_URL não definido. Configure a variável e tente novamente.");
  process.exit(1);
}

const sharedOptions = {
  stdio: "inherit" as const,
  env: {
    ...process.env,
  },
};

try {
  console.log("🏗️  Aplicando schema com drizzle-kit push...");
  execSync("npx drizzle-kit push", sharedOptions);
  console.log("✅ Schema aplicado com sucesso!");
} catch (error) {
  console.error("❌ Erro ao aplicar o schema do banco de dados.");
  process.exit(1);
}

try {
  console.log("🌱 Executando seed.ts...");
  execSync("npx tsx seed.ts", sharedOptions);
  console.log("✅ Seed executado com sucesso!");
} catch (error) {
  console.error("❌ Erro ao executar o seed do banco de dados.");
  process.exit(1);
}

console.log("🎉 Banco de dados preparado com sucesso!");

