#!/usr/bin/env tsx

// Script para preencher thumbVisaoAluno dos cursos existentes
import { resolve } from "path";
import { existsSync } from "fs";
import { config } from "dotenv";

// Carregar variáveis de ambiente do .env ANTES de qualquer import que use DATABASE_URL
const envPath = resolve(process.cwd(), ".env");
if (existsSync(envPath)) {
  config({ path: envPath });
  console.log(`🔐 Variáveis carregadas de ${envPath}`);
} else {
  console.log("⚠️  Arquivo .env não encontrado. Prosseguindo com as variáveis atuais.");
}

if (!process.env.DATABASE_URL) {
  console.error("❌ DATABASE_URL não definido. Configure a variável e tente novamente.");
  process.exit(1);
}

// Importar após carregar .env (usar import dinâmico para garantir que .env foi carregado)
import { courses } from "../shared/schema";
import { eq } from "drizzle-orm";

// Helper function to convert thumbnail URL to 550x550-resized format
function convertThumbnailUrlForStudentView(thumbnailUrl: string | null | undefined): string | null {
  if (!thumbnailUrl) return null;
  
  try {
    // Handle URL with query parameters: https://cdn.areademembros.com/image?p=instancia_308%2Fimage%2FaIJIlLmClVEr3hSBFAddjrLCgJArBnuCiHhUYS5V.png&w=80&h=80&t=crop...
    if (thumbnailUrl.includes('cdn.areademembros.com/image?')) {
      const urlObj = new URL(thumbnailUrl);
      const pParam = urlObj.searchParams.get('p');
      
      if (pParam) {
        // Decode the p parameter: instancia_308%2Fimage%2FaIJIlLmClVEr3hSBFAddjrLCgJArBnuCiHhUYS5V.png
        const decodedPath = decodeURIComponent(pParam);
        // Extract filename: aIJIlLmClVEr3hSBFAddjrLCgJArBnuCiHhUYS5V.png
        const filename = decodedPath.split('/').pop();
        
        if (filename) {
          // Remove extension temporarily
          const nameWithoutExt = filename.replace(/\.(png|jpg|jpeg|gif|webp)$/i, '');
          const ext = filename.match(/\.(png|jpg|jpeg|gif|webp)$/i)?.[1] || 'png';
          
          // Build new URL: https://cdn.areademembros.com/cache/{name}-550x550-resized.{ext}
          return `https://cdn.areademembros.com/cache/${nameWithoutExt}-550x550-resized.${ext}`;
        }
      }
    }
    
    // Handle URL with -80x80-cropped pattern: ...-80x80-cropped.png
    if (thumbnailUrl.includes('-80x80-cropped')) {
      return thumbnailUrl.replace(/-80x80-cropped/g, '-550x550-resized');
    }
    
    // Handle URL with -80x80- pattern (without -cropped): ...-80x80-.png
    if (thumbnailUrl.includes('-80x80-')) {
      return thumbnailUrl.replace(/-80x80-/g, '-550x550-');
    }
    
    // If URL doesn't match any pattern, return as is
    return thumbnailUrl;
  } catch (error) {
    // If URL parsing fails, return original
    console.error('Error converting thumbnail URL:', error);
    return thumbnailUrl;
  }
}

async function migrateThumbVisaoAluno() {
  try {
    console.log("Iniciando migração de thumbVisaoAluno...");
    
    // Importar db dinamicamente após .env estar carregado
    const { db } = await import("../server/db");
    
    // Buscar todos os cursos que têm thumbnailUrl mas não têm thumbVisaoAluno
    const allCourses = await db.select().from(courses);
    
    let updated = 0;
    for (const course of allCourses) {
      // Atualizar se não tem thumbVisaoAluno OU se thumbVisaoAluno é igual ao thumbnailUrl (precisa ser convertido)
      if (course.thumbnailUrl) {
        const thumbVisaoAluno = convertThumbnailUrlForStudentView(course.thumbnailUrl);
        
        // Só atualiza se a conversão resultou em uma URL diferente
        if (thumbVisaoAluno && thumbVisaoAluno !== course.thumbnailUrl) {
          // Verifica se já está atualizado ou se precisa atualizar
          if (!course.thumbVisaoAluno || course.thumbVisaoAluno === course.thumbnailUrl || !course.thumbVisaoAluno.includes('-550x550-resized')) {
            await db
              .update(courses)
              .set({ thumbVisaoAluno })
              .where(eq(courses.id, course.id));
            
            updated++;
            console.log(`✓ Atualizado curso: ${course.title}`);
            console.log(`  De: ${course.thumbnailUrl}`);
            console.log(`  Para: ${thumbVisaoAluno}`);
          }
        }
      }
    }
    
    console.log(`\nMigração concluída! ${updated} curso(s) atualizado(s).`);
    process.exit(0);
  } catch (error) {
    console.error("Erro na migração:", error);
    process.exit(1);
  }
}

migrateThumbVisaoAluno();

