// Student Support Page - Criar tickets de suporte
import { useState, useEffect } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { StudentHeader } from "@/components/student-header";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { MessageSquare, Send, Loader2 } from "lucide-react";
import type { SupportTicket } from "@shared/schema";
import { apiRequest } from "@/lib/queryClient";

export default function StudentSupport() {
  const { isAuthenticated, isLoading, user } = useAuth();
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const [subject, setSubject] = useState("");
  const [message, setMessage] = useState("");
  const [priority, setPriority] = useState<"low" | "medium" | "high" | "urgent">("medium");

  // Redirect if not authenticated
  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  // Fetch user's tickets
  const { data: tickets, isLoading: ticketsLoading } = useQuery<SupportTicket[]>({
    queryKey: ["/api/support/tickets"],
    enabled: isAuthenticated && !!user,
  });

  // Create ticket mutation
  const createTicketMutation = useMutation({
    mutationFn: async (data: { subject: string; message: string; priority: string }) => {
      const response = await apiRequest("POST", "/api/support/tickets", data);
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Ticket criado",
        description: "Seu ticket de suporte foi criado com sucesso!",
      });
      setSubject("");
      setMessage("");
      setPriority("medium");
      queryClient.invalidateQueries({ queryKey: ["/api/support/tickets"] });
    },
    onError: (error: any) => {
      toast({
        variant: "destructive",
        title: "Erro",
        description: error.message || "Erro ao criar ticket de suporte",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!subject.trim() || !message.trim()) {
      toast({
        variant: "destructive",
        title: "Campos obrigatórios",
        description: "Preencha todos os campos para criar o ticket",
      });
      return;
    }
    createTicketMutation.mutate({ subject, message, priority });
  };

  const getStatusVariant = (status: string): "default" | "secondary" | "outline" => {
    switch (status) {
      case "open":
        return "default";
      case "in_progress":
        return "secondary";
      case "resolved":
        return "outline";
      default:
        return "secondary";
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case "urgent":
        return "text-destructive";
      case "high":
        return "text-chart-4";
      case "medium":
        return "text-chart-1";
      default:
        return "text-muted-foreground";
    }
  };

  if (isLoading || ticketsLoading) {
    return (
      <div className="min-h-screen bg-background" style={{ backgroundColor: '#121212' }}>
        <StudentHeader />
        <div className="pt-16 sm:pt-20 animate-pulse">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 py-6 sm:py-12">
            <div className="h-6 sm:h-8 bg-muted rounded w-48 sm:w-64 mb-6 sm:mb-8"></div>
            <div className="h-64 sm:h-96 bg-muted rounded"></div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background" style={{ backgroundColor: '#121212' }}>
      <StudentHeader />
      
      <div className="pt-16 sm:pt-20">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 py-6 sm:py-12">
          <div className="flex items-center gap-2 sm:gap-3 mb-6 sm:mb-8">
            <MessageSquare className="w-6 h-6 sm:w-8 sm:h-8 text-white shrink-0" />
            <h1 className="text-2xl sm:text-3xl md:text-4xl font-serif font-semibold text-white">
              Suporte
            </h1>
          </div>

          <Tabs defaultValue="new" className="space-y-6">
            <TabsList>
              <TabsTrigger value="new">
                <Send className="w-4 h-4 mr-2" />
                Novo Ticket
              </TabsTrigger>
              <TabsTrigger value="tickets">
                <MessageSquare className="w-4 h-4 mr-2" />
                Meus Tickets
              </TabsTrigger>
            </TabsList>

            <TabsContent value="new">
              <Card className="bg-card border-border">
                <CardHeader>
                  <CardTitle className="text-base sm:text-lg">Criar novo ticket de suporte</CardTitle>
                  <CardDescription className="text-sm">
                    Descreva seu problema ou dúvida e nossa equipe entrará em contato
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <form onSubmit={handleSubmit} className="space-y-3 sm:space-y-4">
                    <div className="space-y-2">
                      <Label htmlFor="subject">Assunto</Label>
                      <Input
                        id="subject"
                        value={subject}
                        onChange={(e) => setSubject(e.target.value)}
                        placeholder="Ex: Problema ao acessar curso"
                        required
                      />
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="priority">Prioridade</Label>
                      <Select value={priority} onValueChange={(value: any) => setPriority(value)}>
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="low">Baixa</SelectItem>
                          <SelectItem value="medium">Média</SelectItem>
                          <SelectItem value="high">Alta</SelectItem>
                          <SelectItem value="urgent">Urgente</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div className="space-y-2">
                      <Label htmlFor="message">Mensagem</Label>
                      <Textarea
                        id="message"
                        value={message}
                        onChange={(e) => setMessage(e.target.value)}
                        placeholder="Descreva seu problema ou dúvida em detalhes..."
                        rows={6}
                        required
                      />
                    </div>

                    <Button
                      type="submit"
                      disabled={createTicketMutation.isPending}
                      className="w-full"
                    >
                      {createTicketMutation.isPending ? (
                        <>
                          <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                          Enviando...
                        </>
                      ) : (
                        <>
                          <Send className="w-4 h-4 mr-2" />
                          Enviar Ticket
                        </>
                      )}
                    </Button>
                  </form>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="tickets">
              <Card className="bg-card border-border">
                <CardHeader>
                  <CardTitle className="text-base sm:text-lg">Meus Tickets de Suporte</CardTitle>
                  <CardDescription className="text-sm">
                    Acompanhe o status dos seus tickets
                  </CardDescription>
                </CardHeader>
                <CardContent className="p-0">
                  <div className="overflow-x-auto">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead className="min-w-[150px]">Assunto</TableHead>
                          <TableHead className="min-w-[100px]">Prioridade</TableHead>
                          <TableHead className="min-w-[100px]">Status</TableHead>
                          <TableHead className="min-w-[100px]">Data</TableHead>
                        </TableRow>
                      </TableHeader>
                    <TableBody>
                      {tickets?.filter(ticket => ticket.userId === user?.id).map((ticket) => (
                        <TableRow key={ticket.id}>
                          <TableCell className="font-medium">{ticket.subject}</TableCell>
                          <TableCell>
                            <span className={`text-sm font-medium ${getPriorityColor(ticket.priority)}`}>
                              {ticket.priority === "urgent" ? "Urgente" :
                               ticket.priority === "high" ? "Alta" :
                               ticket.priority === "medium" ? "Média" : "Baixa"}
                            </span>
                          </TableCell>
                          <TableCell>
                            <Badge variant={getStatusVariant(ticket.status)}>
                              {ticket.status === "open"
                                ? "Aberto"
                                : ticket.status === "in_progress"
                                ? "Em Andamento"
                                : ticket.status === "resolved"
                                ? "Resolvido"
                                : "Fechado"}
                            </Badge>
                          </TableCell>
                          <TableCell className="text-muted-foreground text-sm">
                            {ticket.createdAt ? new Date(ticket.createdAt).toLocaleDateString("pt-BR") : "-"}
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                    </Table>
                  </div>
                  {(!tickets || tickets.filter(t => t.userId === user?.id).length === 0) && (
                    <div className="text-center py-12 text-muted-foreground">
                      <MessageSquare className="w-12 h-12 mx-auto mb-4 opacity-50" />
                      <p>Você ainda não criou nenhum ticket de suporte</p>
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </div>
  );
}

