// Student FAQ Page - Perguntas Frequentes
import { useEffect } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useQuery } from "@tanstack/react-query";
import { StudentHeader } from "@/components/student-header";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import {
  Accordion,
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from "@/components/ui/accordion";
import { HelpCircle } from "lucide-react";
import type { FaqItem } from "@shared/schema";

export default function StudentFAQ() {
  const { isAuthenticated, isLoading } = useAuth();
  const { toast } = useToast();

  // Redirect if not authenticated
  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: faqs, isLoading: dataLoading } = useQuery<FaqItem[]>({
    queryKey: ["/api/faq"],
    enabled: isAuthenticated,
  });

  if (isLoading || dataLoading) {
    return (
      <div className="min-h-screen bg-background" style={{ backgroundColor: '#121212' }}>
        <StudentHeader />
        <div className="pt-16 sm:pt-20 animate-pulse">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 py-6 sm:py-12">
            <div className="h-6 sm:h-8 bg-muted rounded w-48 sm:w-64 mb-6 sm:mb-8"></div>
            <div className="space-y-3 sm:space-y-4">
              {[1, 2, 3, 4, 5].map((i) => (
                <div key={i} className="h-20 sm:h-24 bg-muted rounded"></div>
              ))}
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background" style={{ backgroundColor: '#121212' }}>
      <StudentHeader />
      
      <div className="pt-16 sm:pt-20">
        <div className="max-w-4xl mx-auto px-4 sm:px-6 py-6 sm:py-12">
          <div className="flex items-center gap-2 sm:gap-3 mb-6 sm:mb-8">
            <HelpCircle className="w-6 h-6 sm:w-8 sm:h-8 text-white shrink-0" />
            <h1 className="text-2xl sm:text-3xl md:text-4xl font-serif font-semibold text-white">
              Perguntas Frequentes
            </h1>
          </div>

          <Card className="bg-card border-border">
            <CardHeader>
              <CardTitle className="text-base sm:text-lg">Encontre respostas para suas dúvidas</CardTitle>
            </CardHeader>
            <CardContent>
              {faqs && faqs.length > 0 ? (
                <Accordion type="single" collapsible className="w-full">
                  {faqs.map((faq, index) => (
                    <AccordionItem 
                      key={faq.id} 
                      value={`item-${index}`}
                    >
                      <AccordionTrigger className="text-left">
                        {faq.question}
                      </AccordionTrigger>
                      <AccordionContent className="text-muted-foreground">
                        {faq.answer}
                      </AccordionContent>
                    </AccordionItem>
                  ))}
                </Accordion>
              ) : (
                <div className="text-center py-12 text-muted-foreground">
                  <HelpCircle className="w-12 h-12 mx-auto mb-4 opacity-50" />
                  <p>Nenhuma pergunta disponível no momento.</p>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}

