// Student Dashboard - Vitrines com cursos
import { useEffect } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { StudentHeader } from "@/components/student-header";
import { CourseCard } from "@/components/course-card";
import { Play } from "lucide-react";
import type { Course } from "@shared/schema";

interface Vitrine {
  id: string;
  name: string;
  description: string | null;
  orderIndex: number;
  courses: Course[];
}

export default function StudentDashboard() {
  const { isAuthenticated, isLoading } = useAuth();
  const { toast } = useToast();
  const [, setLocation] = useLocation();

  // Redirect if not authenticated
  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: vitrines, isLoading: dataLoading } = useQuery<Vitrine[]>({
    queryKey: ["/api/student/vitrines"],
    enabled: isAuthenticated,
  });

  if (isLoading || dataLoading) {
    return (
      <div className="min-h-screen bg-background">
        <StudentHeader />
        <div className="pt-16 sm:pt-20 animate-pulse">
          <div className="max-w-screen-2xl mx-auto px-4 sm:px-6 py-6 sm:py-12 space-y-8 sm:space-y-12">
            <div className="h-6 sm:h-8 bg-muted rounded w-48 sm:w-64"></div>
            <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-6 gap-3 sm:gap-4">
              {[1, 2, 3, 4, 5, 6].map((i) => (
                <div key={i} className="h-48 sm:h-64 md:h-80 bg-muted rounded"></div>
              ))}
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      <StudentHeader />
      
      {/* Vitrines Section */}
      <div className="pt-16 sm:pt-20">
        <div className="max-w-screen-2xl mx-auto px-4 sm:px-6 py-6 sm:py-12 space-y-8 sm:space-y-12">
          {vitrines && vitrines.length > 0 ? (
            vitrines.map((vitrine) => (
              <div key={vitrine.id} className="space-y-4 sm:space-y-6">
                {/* Nome da Vitrine */}
                <h2 className="text-xl sm:text-2xl md:text-3xl font-serif font-semibold text-foreground">
                  {vitrine.name}
                </h2>
                
                {/* Thumbnails dos Cursos */}
                {vitrine.courses.length > 0 ? (
                  <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 xl:grid-cols-6 gap-3 sm:gap-4">
                    {vitrine.courses.map((course) => (
                      <CourseCard
                        key={course.id}
                        course={course}
                        onClick={() => setLocation(`/course/${course.id}`)}
                      />
                    ))}
                  </div>
                ) : (
                  <div className="text-center py-6 sm:py-8 text-muted-foreground">
                    <p className="text-sm sm:text-base">Nenhum curso disponível nesta vitrine.</p>
                  </div>
                )}
              </div>
            ))
          ) : (
            <div className="text-center py-12 sm:py-24 px-4">
              <Play className="w-12 h-12 sm:w-16 sm:h-16 mx-auto mb-4 text-muted-foreground opacity-50" />
              <h3 className="text-xl sm:text-2xl font-serif font-semibold mb-2">Nenhuma vitrine disponível</h3>
              <p className="text-sm sm:text-base text-muted-foreground">Entre em contato com o administrador para obter acesso aos cursos.</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}
