// Landing Page - Email/Password Login for VPS deployment
import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { GraduationCap, TrendingUp, Users, Award, Loader2 } from "lucide-react";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import logoUrl from "@assets/logo.png";

export default function Landing() {
  const [mode, setMode] = useState<"login" | "register">("login");
  const [email, setEmail] = useState("");
  const [password, setPassword] = useState("");
  const [firstName, setFirstName] = useState("");
  const [lastName, setLastName] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const { toast } = useToast();

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);

    try {
      const endpoint = mode === "login" ? "/api/login" : "/api/register";
      const body = mode === "login" 
        ? { email, password }
        : { email, password, firstName, lastName };

      const response = await fetch(endpoint, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(body),
        credentials: "include",
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.message || "Erro ao processar solicitação");
      }

      // Invalidate queries and reload
      await queryClient.invalidateQueries();
      window.location.href = "/";
    } catch (error: any) {
      toast({
        variant: "destructive",
        title: "Erro",
        description: error.message || `Erro ao ${mode === "login" ? "fazer login" : "registrar"}`,
      });
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <div className="min-h-screen grid lg:grid-cols-2">
      {/* Left Panel - Brand */}
      <div className="hidden lg:flex flex-col justify-center items-center p-12 bg-gradient-to-br from-primary via-primary/90 to-primary/70 text-primary-foreground">
        <div className="max-w-md space-y-8">
          <div className="flex items-center gap-3">
            <img 
              src={logoUrl} 
              alt="Portal de Cursos" 
              className="h-16 w-auto object-contain"
            />
            <h1 className="font-serif font-bold text-4xl">Portal de Cursos</h1>
          </div>
          
          <p className="text-xl text-primary-foreground/90 leading-relaxed">
            Transforme seu aprendizado em resultados concretos. Acesse cursos de alta qualidade e impulsione sua carreira.
          </p>

          <div className="space-y-4 pt-4">
            <div className="flex items-center gap-3">
              <div className="w-12 h-12 rounded-md bg-primary-foreground/10 flex items-center justify-center">
                <Users className="w-6 h-6" />
              </div>
              <div>
                <p className="font-semibold text-lg">10.000+ Alunos</p>
                <p className="text-sm text-primary-foreground/70">Estudando conosco</p>
              </div>
            </div>
            
            <div className="flex items-center gap-3">
              <div className="w-12 h-12 rounded-md bg-primary-foreground/10 flex items-center justify-center">
                <Award className="w-6 h-6" />
              </div>
              <div>
                <p className="font-semibold text-lg">Certificados Reconhecidos</p>
                <p className="text-sm text-primary-foreground/70">Valide seu conhecimento</p>
              </div>
            </div>
            
            <div className="flex items-center gap-3">
              <div className="w-12 h-12 rounded-md bg-primary-foreground/10 flex items-center justify-center">
                <TrendingUp className="w-6 h-6" />
              </div>
              <div>
                <p className="font-semibold text-lg">Aprenda no Seu Ritmo</p>
                <p className="text-sm text-primary-foreground/70">Conteúdo sempre disponível</p>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Right Panel - Login/Register Form */}
      <div className="flex flex-col justify-center items-center p-4 sm:p-6 lg:p-8 bg-background">
        <div className="w-full max-w-md">
          {/* Mobile Logo */}
          <div className="text-center lg:hidden mb-6 sm:mb-8">
            <img 
              src={logoUrl} 
              alt="Portal de Cursos" 
              className="h-12 sm:h-16 w-auto mx-auto mb-3 sm:mb-4 object-contain"
            />
            <h1 className="font-serif font-bold text-2xl sm:text-3xl mb-2">Portal de Cursos</h1>
            <p className="text-sm sm:text-base text-muted-foreground">Acesse sua conta para continuar</p>
          </div>

          <Card>
            <CardHeader>
              <CardTitle className="font-serif text-xl sm:text-2xl">
                {mode === "login" ? "Bem-vindo!" : "Criar Conta"}
              </CardTitle>
              <CardDescription className="text-sm">
                {mode === "login"
                  ? "Entre com seu email e senha"
                  : "Preencha os dados para criar sua conta"}
              </CardDescription>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-3 sm:space-y-4">
                {mode === "register" && (
                  <div className="grid grid-cols-2 gap-3 sm:gap-4">
                    <div className="space-y-2">
                      <Label htmlFor="firstName">Nome</Label>
                      <Input
                        id="firstName"
                        type="text"
                        value={firstName}
                        onChange={(e) => setFirstName(e.target.value)}
                        placeholder="João"
                        data-testid="input-firstName"
                      />
                    </div>
                    <div className="space-y-2">
                      <Label htmlFor="lastName">Sobrenome</Label>
                      <Input
                        id="lastName"
                        type="text"
                        value={lastName}
                        onChange={(e) => setLastName(e.target.value)}
                        placeholder="Silva"
                        data-testid="input-lastName"
                      />
                    </div>
                  </div>
                )}

                <div className="space-y-2">
                  <Label htmlFor="email">Email</Label>
                  <Input
                    id="email"
                    type="email"
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    placeholder="seu@email.com"
                    required
                    data-testid="input-email"
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="password">Senha</Label>
                  <Input
                    id="password"
                    type="password"
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    placeholder="••••••••"
                    required
                    minLength={6}
                    data-testid="input-password"
                  />
                  {mode === "register" && (
                    <p className="text-xs text-muted-foreground">
                      Mínimo 6 caracteres
                    </p>
                  )}
                </div>

                <Button
                  type="submit"
                  className="w-full"
                  disabled={isLoading}
                  data-testid="button-submit"
                >
                  {isLoading ? (
                    <>
                      <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                      Processando...
                    </>
                  ) : mode === "login" ? (
                    "Entrar"
                  ) : (
                    "Criar Conta"
                  )}
                </Button>
              </form>

              <div className="mt-4 text-center">
                <button
                  type="button"
                  onClick={() => setMode(mode === "login" ? "register" : "login")}
                  className="text-sm text-primary hover:underline"
                  data-testid="button-toggle-mode"
                >
                  {mode === "login"
                    ? "Não tem uma conta? Registre-se"
                    : "Já tem uma conta? Faça login"}
                </button>
              </div>

              <div className="mt-6 pt-6 border-t">
                <div className="flex items-center justify-center gap-8 text-center text-sm">
                  <div>
                    <p className="text-lg font-bold font-mono text-primary">10k+</p>
                    <p className="text-xs text-muted-foreground">Alunos</p>
                  </div>
                  <div className="h-8 w-px bg-border"></div>
                  <div>
                    <p className="text-lg font-bold font-mono text-chart-2">500+</p>
                    <p className="text-xs text-muted-foreground">Cursos</p>
                  </div>
                  <div className="h-8 w-px bg-border"></div>
                  <div>
                    <p className="text-lg font-bold font-mono text-chart-3">98%</p>
                    <p className="text-xs text-muted-foreground">Satisfação</p>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
