// Course Player - Redesigned based on student view requirements
import { useState, useEffect, useRef } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useQuery, useMutation } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { StudentHeader } from "@/components/student-header";
import { Play, CheckCircle, ChevronRight, Circle, ChevronDown } from "lucide-react";
import { useLocation, Link } from "wouter";

// Helper function to strip HTML tags and format text
function stripHtmlTags(html: string): string {
  if (!html) return "";
  // Remove HTML tags
  let text = html.replace(/<[^>]*>/g, "");
  // Decode HTML entities
  text = text
    .replace(/&nbsp;/g, " ")
    .replace(/&amp;/g, "&")
    .replace(/&lt;/g, "<")
    .replace(/&gt;/g, ">")
    .replace(/&quot;/g, '"')
    .replace(/&#39;/g, "'")
    .replace(/&apos;/g, "'");
  return text;
}

// Helper function to extract PandaVideo ID from M3U8 URL
function extractPandaVideoId(videoUrl: string | null): string | null {
  if (!videoUrl) return null;
  
  // Check if it's a PandaVideo M3U8 URL
  // Pattern: https://b-vz-xxxxx.tv.pandavideo.com.br/VIDEO_ID/playlist.m3u8
  const pandaVideoMatch = videoUrl.match(/pandavideo\.com\.br\/([a-f0-9-]+)\/playlist\.m3u8/i);
  if (pandaVideoMatch && pandaVideoMatch[1]) {
    return pandaVideoMatch[1];
  }
  
  return null;
}

// Helper function to check if video URL is from PandaVideo
function isPandaVideoUrl(videoUrl: string | null): boolean {
  return extractPandaVideoId(videoUrl) !== null;
}

// Helper function to format text with proper line breaks and structure
function formatText(text: string): string {
  if (!text) return "";
  let formatted = stripHtmlTags(text);
  
  // Preserve existing line breaks
  formatted = formatted.replace(/\r\n/g, "\n").replace(/\r/g, "\n");
  
  // IMPORTANT: Split items separated by bullet (•) into separate lines
  // Handle both "texto • texto" (with spaces) and "texto•texto" (without spaces)
  // Split on bullet character, with optional spaces around it
  // This will convert "Item1•Item2•Item3" into separate lines
  formatted = formatted.replace(/([^\s\n])([•])/g, "$1\n$2"); // Add newline before bullet if no space
  formatted = formatted.replace(/([•])([^\s\n])/g, "$1 $2"); // Add space after bullet if missing
  formatted = formatted.replace(/\s*([•])\s*/g, "\n$1 "); // Normalize: bullet with space, on new line
  
  // IMPORTANT: Split timestamps into separate lines
  // Pattern: "00:06 – texto" or "00:06 - texto" should be on its own line
  // Add line break before timestamp pattern
  formatted = formatted.replace(/([^\n])(\d{1,2}:\d{2}\s*[–-])/g, "$1\n$2");
  
  // Add line breaks before section headers (all caps followed by colon, like "CARGA HORÁRIA:")
  formatted = formatted.replace(/([a-z0-9]) ([A-ZÁÉÍÓÚÇ]{2,}:)/g, "$1\n\n$2");
  
  // Add line breaks after section headers (colon followed by content)
  formatted = formatted.replace(/([A-ZÁÉÍÓÚÇ]{2,}:) ([A-Z])/g, "$1\n$2");
  
  // Format bullet points (• or -) - add line break before, ensure space after
  formatted = formatted.replace(/([^\n]) ([•\-])/g, "$1\n$2");
  formatted = formatted.replace(/([•\-])([^\s\n])/g, "$1 $2");
  
  // Add line breaks after sentences ending with period, exclamation, or question mark
  // Followed by space and capital letter (new sentence) - but only if it's a new paragraph
  // This helps separate paragraphs in video lesson descriptions
  formatted = formatted.replace(/([.!?])\s+([A-ZÁÉÍÓÚÇ][a-záéíóúç][^.!?]{20,})/g, "$1\n\n$2");
  
  // Add line breaks after periods followed by capital (but not abbreviations or short sentences)
  formatted = formatted.replace(/\. ([A-ZÁÉÍÓÚÇ][a-záéíóúç])/g, ".\n$1");
  
  // Special handling for common patterns that should start new paragraphs
  formatted = formatted.replace(/([.!?]) (Seja muito bem-vindo)/g, "$1\n\n$2");
  formatted = formatted.replace(/([.!?]) (Conversamos sobre)/g, "$1\n\n$2");
  formatted = formatted.replace(/([.!?]) (Olá!)/g, "$1\n\n$2");
  
  // Add line breaks before common section patterns (even if not followed by colon yet)
  formatted = formatted.replace(/([a-z0-9]) (CARGA HORÁRIA)/g, "$1\n\n$2");
  formatted = formatted.replace(/([a-z0-9]) (CONTEÚDO PROGRAMÁTICO)/g, "$1\n\n$2");
  
  // Ensure section headers are on their own line
  formatted = formatted.replace(/(CARGA HORÁRIA:)/g, "\n\n$1");
  formatted = formatted.replace(/(CONTEÚDO PROGRAMÁTICO:)/g, "\n\n$1");
  
  // Clean up multiple spaces (but keep single spaces)
  formatted = formatted.replace(/  +/g, " ");
  
  // Clean up multiple newlines (max 2 consecutive)
  formatted = formatted.replace(/\n{3,}/g, "\n\n");
  
  // Clean up spaces at start of lines
  formatted = formatted.replace(/\n +/g, "\n");
  
  // Clean up spaces at end of lines
  formatted = formatted.replace(/ +\n/g, "\n");
  
  return formatted.trim();
}

interface CourseData {
  course: {
    id: string;
    title: string;
    description: string | null;
    instructorName: string | null;
    instructorBio: string | null;
  };
  modules: Array<{
    id: string;
    title: string;
    description: string | null;
    orderIndex: number;
    lessons: Array<{
      id: string;
      title: string;
      description: string | null;
      videoUrl: string | null;
      videoDuration: number | null;
      orderIndex: number;
      completed: boolean;
      progressPercentage: number;
    }>;
  }>;
  materials: Array<{
    id: string;
    title: string;
    fileName: string;
    fileMimeType: string;
    fileSize: number | null;
  }>;
  overallProgress: number;
}

interface CoursePlayerProps {
  params: { id: string };
}

export default function CoursePlayer({ params }: CoursePlayerProps) {
  const { isAuthenticated, isLoading } = useAuth();
  const { toast } = useToast();
  const [, setLocation] = useLocation();
  const [currentLessonId, setCurrentLessonId] = useState<string | null>(null);
  const [expandedModuleId, setExpandedModuleId] = useState<string | null>(null);
  const [isPlaying, setIsPlaying] = useState(false);
  const videoRef = useRef<HTMLVideoElement>(null);
  const progressIntervalRef = useRef<NodeJS.Timeout | null>(null);
  const autoCompletedLessonsRef = useRef<Set<string>>(new Set());

  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: courseData, isLoading: dataLoading } = useQuery<CourseData>({
    queryKey: ["/api/student/course", params.id],
    enabled: isAuthenticated,
  });

  const markCompleteMutation = useMutation({
    mutationFn: async (lessonId: string) => {
      return await apiRequest("POST", "/api/student/progress", { lessonId, completed: true });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/student/course", params.id] });
      toast({ title: "Aula marcada como concluída!" });
    },
  });

  const updateProgressMutation = useMutation({
    mutationFn: async ({ lessonId, progressPercentage }: { lessonId: string; progressPercentage: number }) => {
      return await apiRequest("POST", "/api/student/progress", { lessonId, progressPercentage, completed: false });
    },
  });

  // Get current lesson
  const currentLesson = courseData?.modules
    .flatMap(m => m.lessons)
    .find(l => l.id === currentLessonId);

  // Check if lesson has video
  const hasVideo = (() => {
    if (!currentLesson) return false;
    
    // Check if lesson title indicates it's text-only (Apresentação, Encerramento)
    const titleLower = currentLesson.title.toLowerCase();
    if (titleLower.includes("apresentação") || titleLower.includes("encerramento")) {
      return false;
    }
    
    // Check if lesson has videoUrl or videoDuration (indicates video exists)
    // videoDuration is a strong indicator that video exists
    // videoUrl also indicates video exists (external URL)
    return !!(currentLesson.videoUrl || currentLesson.videoDuration);
  })();

  // Não auto-seleciona a primeira aula - o aluno deve clicar no módulo primeiro

  // Track video progress periodically (only for non-PandaVideo videos)
  useEffect(() => {
    if (!currentLessonId || !videoRef.current || !currentLesson || !hasVideo) return;
    
    // Skip progress tracking for PandaVideo (iframe doesn't expose video events)
    if (currentLesson.videoUrl && isPandaVideoUrl(currentLesson.videoUrl)) {
      return;
    }

    if (currentLesson.completed) {
      if (progressIntervalRef.current) {
        clearInterval(progressIntervalRef.current);
      }
      return;
    }

    const updateProgress = () => {
      if (!videoRef.current) return;
      
      const duration = videoRef.current.duration;
      const currentTime = videoRef.current.currentTime;
      
      if (!duration || isNaN(duration) || duration === 0) return;
      
      const percentage = Math.round((currentTime / duration) * 100);
      
      if (!isNaN(percentage) && percentage > 0 && percentage <= 100) {
        updateProgressMutation.mutate({ lessonId: currentLessonId, progressPercentage: percentage });
      }

      if (
        percentage >= 95 &&
        !autoCompletedLessonsRef.current.has(currentLessonId)
      ) {
        autoCompletedLessonsRef.current.add(currentLessonId);
        markCompleteMutation.mutate(currentLessonId);
      }
    };

    if (isPlaying) {
      progressIntervalRef.current = setInterval(updateProgress, 10000);
    }

    return () => {
      if (progressIntervalRef.current) {
        clearInterval(progressIntervalRef.current);
      }
    };
  }, [currentLessonId, isPlaying, currentLesson?.completed, hasVideo, currentLesson?.videoUrl]);

  // Reset video when lesson changes (only for non-PandaVideo videos)
  useEffect(() => {
    setIsPlaying(false);
    // Only reset if it's not a PandaVideo (iframe handles its own state)
    if (videoRef.current && currentLesson && currentLesson.videoUrl && !isPandaVideoUrl(currentLesson.videoUrl)) {
      videoRef.current.load();
    }
  }, [currentLessonId, currentLesson?.videoUrl]);

  // Calculate total hours (approximate from video durations)
  const totalHours = courseData?.modules
    .flatMap(m => m.lessons)
    .reduce((acc, lesson) => acc + (lesson.videoDuration || 0), 0) || 0;
  const totalHoursFormatted = Math.round(totalHours / 3600);

  // Get all lesson titles for programmatic content
  const programmaticContent = courseData?.modules.flatMap(m => 
    m.lessons.map(l => l.title)
  ) || [];

  const totalLessons = courseData?.modules.flatMap(m => m.lessons).length || 0;
  const completedLessons = courseData?.modules.flatMap(m => m.lessons).filter(l => l.completed).length || 0;

  if (isLoading || dataLoading) {
    return (
      <div className="min-h-screen bg-background">
        <StudentHeader />
        <div className="pt-16 sm:pt-20 animate-pulse space-y-4 max-w-7xl mx-auto p-4 sm:p-6">
          <div className="h-48 sm:h-64 md:h-96 bg-muted rounded-md"></div>
          <div className="h-24 sm:h-32 bg-muted rounded-md"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      <StudentHeader />
      
      <div className="pt-16 sm:pt-20">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 py-4 sm:py-6 lg:py-8">
          {/* Breadcrumbs */}
          <nav className="mb-4 sm:mb-6 text-xs sm:text-sm text-muted-foreground">
            <Link href="/" className="hover:text-primary transition-colors">Início</Link>
            <span className="mx-1 sm:mx-2">/</span>
            <span className="text-foreground truncate">{courseData?.course.title}</span>
            {currentLesson && (
              <>
                <span className="mx-1 sm:mx-2">/</span>
                                  <Link 
                  href="#" 
                  className="text-primary hover:underline"
                  onClick={(e) => {
                    e.preventDefault();
                    // Could add edit functionality here
                  }}
                >
                  Editar esse conteúdo
                </Link>
              </>
            )}
          </nav>

          <div className="grid grid-cols-1 lg:grid-cols-3 gap-4 sm:gap-6 lg:gap-8">
            {/* Main Content - Left Side */}
            <div className="lg:col-span-2 space-y-4 sm:space-y-6">
              {/* Course Title */}
              <h1 className="text-xl sm:text-2xl md:text-3xl lg:text-4xl font-serif font-semibold text-foreground">
                {currentLesson 
                  ? `${currentLesson.title} - ${courseData?.course.title}`
                  : `Apresentação - ${courseData?.course.title}`
                }
              </h1>

              {/* Video Player - Only show when lesson has video */}
              {currentLessonId && hasVideo && (
                <div className="aspect-video bg-black rounded-lg overflow-hidden relative">
                  {currentLesson?.videoUrl && isPandaVideoUrl(currentLesson.videoUrl) ? (
                    // PandaVideo - use iframe
                    (() => {
                      const videoId = extractPandaVideoId(currentLesson.videoUrl);
                      if (!videoId) return null;
                      
                      // Extract the subdomain from the original URL to build the player URL
                      // Pattern: https://b-vz-xxxxx.tv.pandavideo.com.br/... -> player-vz-xxxxx
                      const subdomainMatch = currentLesson.videoUrl.match(/https?:\/\/([^\/]+)\.tv\.pandavideo\.com\.br/i);
                      let playerSubdomain = 'player-vz-85544a04-b57'; // default fallback
                      
                      if (subdomainMatch) {
                        const originalSubdomain = subdomainMatch[1];
                        // Convert b-vz-xxxxx to player-vz-xxxxx
                        if (originalSubdomain.startsWith('b-vz-')) {
                          playerSubdomain = originalSubdomain.replace('b-vz-', 'player-vz-');
                        } else {
                          // If it's already player-vz- or different format, use as is
                          playerSubdomain = originalSubdomain.startsWith('player-') 
                            ? originalSubdomain 
                            : `player-${originalSubdomain}`;
                        }
                      }
                      
                      return (
                        <iframe
                          id="panda-video"
                          style={{ width: "100%", height: "100%", position: "absolute", top: 0, left: 0 }}
                          src={`https://${playerSubdomain}.tv.pandavideo.com.br/embed/?v=${videoId}`}
                          title="Video player"
                          frameBorder="0"
                          allow="accelerometer; gyroscope; autoplay; encrypted-media; picture-in-picture"
                          allowFullScreen
                          fetchPriority="high"
                        />
                      );
                    })()
                  ) : currentLesson?.videoUrl ? (
                    // External video URL (non-PandaVideo) - use directly
                    <video
                      ref={videoRef}
                      className="w-full h-full"
                      controls
                      controlsList="nodownload"
                      onPlay={() => setIsPlaying(true)}
                      onPause={() => setIsPlaying(false)}
                      onEnded={() => {
                        setIsPlaying(false);
                        if (currentLesson && !currentLesson.completed) {
                          markCompleteMutation.mutate(currentLessonId);
                        }
                      }}
                      onError={(e) => {
                        console.error("Video error:", e);
                        toast({
                          title: "Erro ao carregar vídeo",
                          description: "O vídeo não pôde ser carregado.",
                          variant: "destructive",
                        });
                      }}
                      data-testid="video-player"
                    >
                      <source src={currentLesson.videoUrl} type="video/mp4" />
                      Seu navegador não suporta reprodução de vídeo.
                    </video>
                  ) : (
                    // Video from API endpoint (base64 stored in database)
                    <video
                      ref={videoRef}
                      className="w-full h-full"
                      controls
                      controlsList="nodownload"
                      onPlay={() => setIsPlaying(true)}
                      onPause={() => setIsPlaying(false)}
                      onEnded={() => {
                        setIsPlaying(false);
                        if (currentLesson && !currentLesson.completed) {
                          markCompleteMutation.mutate(currentLessonId);
                        }
                      }}
                      onError={(e) => {
                        console.error("Video error:", e);
                        toast({
                          title: "Erro ao carregar vídeo",
                          description: "O vídeo não pôde ser carregado. Verifique sua conexão.",
                          variant: "destructive",
                        });
                      }}
                      data-testid="video-player"
                    >
                      <source
                        src={`/api/student/lesson/${currentLessonId}/video`}
                        type="video/mp4"
                      />
                      Seu navegador não suporta reprodução de vídeo.
                    </video>
                  )}
                </div>
              )}

              {/* Course Introduction - Show when no lesson selected */}
              {!currentLessonId && (
                <div className="space-y-6 text-foreground">
                  {/* Introduction Text */}
                  {courseData?.course.description && (
                    <div className="space-y-4">
                      <p className="text-foreground leading-relaxed whitespace-pre-wrap text-sm sm:text-base">
                        {formatText(courseData.course.description)}
                      </p>
                    </div>
                  )}

                  {/* Workload */}
                  {totalHoursFormatted > 0 && (
                    <div>
                      <p className="text-foreground font-semibold text-sm sm:text-base">CARGA HORÁRIA: {totalHoursFormatted} horas/aula</p>
                    </div>
                  )}

                  {/* Programmatic Content */}
                  {programmaticContent.length > 0 && (
                    <div>
                      <h3 className="text-foreground font-semibold mb-3 sm:mb-4 text-sm sm:text-base">CONTEÚDO PROGRAMÁTICO:</h3>
                      <ul className="space-y-2 list-disc list-inside text-foreground text-sm sm:text-base">
                        {programmaticContent.map((title, index) => (
                          <li key={index}>{title}</li>
                        ))}
                      </ul>
                    </div>
                  )}

                  {/* Instructor Info */}
                  {courseData?.course.instructorName && (
                    <div className="pt-4 border-t border-border">
                      <p className="text-foreground">
                        {courseData.course.instructorBio 
                          ? `Conversamos sobre esse tema com a instrutora ${courseData.course.instructorName}.`
                          : `Instrutor: ${courseData.course.instructorName}`
                        }
                      </p>
                      {courseData.course.instructorBio && (
                        <p className="text-muted-foreground mt-2">{formatText(courseData.course.instructorBio)}</p>
                      )}
                    </div>
                  )}

                  {/* Welcome Message */}
                  <div className="pt-4">
                    <p className="text-foreground">
                      Seja muito bem-vindo ao curso: {courseData?.course.title}.
                    </p>
                  </div>
                </div>
              )}

              {/* Lesson Content - Show when lesson is selected */}
              {currentLesson && (
                <div className="space-y-6">
                  {/* For text-only lessons (Apresentação, Encerramento) */}
                  {!hasVideo && currentLesson.description && (
                    <div className="text-foreground leading-relaxed space-y-6">
                      {(() => {
                        const formatted = formatText(currentLesson.description);
                        const lines = formatted.split("\n");
                        const elements: JSX.Element[] = [];
                        let currentList: string[] = [];
                        let inList = false;
                        let lastWasList = false;
                        
                        lines.forEach((line, index) => {
                          const trimmed = line.trim();
                          if (!trimmed) {
                            // Empty line - close list if open and add spacing
                            if (inList && currentList.length > 0) {
                              elements.push(
                                <ul key={`list-${index}`} className="list-disc list-inside space-y-2 ml-6 mb-6">
                                  {currentList.map((item, i) => (
                                    <li key={i} className="mb-1.5">{item.replace(/^[•\-]\s*/, "").trim()}</li>
                                  ))}
                                </ul>
                              );
                              currentList = [];
                              inList = false;
                              lastWasList = true;
                            } else if (lastWasList) {
                              // Add extra spacing after list
                              lastWasList = false;
                            }
                            return;
                          }
                          
                          // FIRST: Check if line contains multiple items separated by bullet (•)
                          // This must be checked BEFORE checking if it starts with bullet
                          // Handle both "texto • texto" and "texto•texto" patterns
                          if (trimmed.includes("•") && (trimmed.match(/•/g) || []).length > 0) {
                            // Close current list if open
                            if (inList && currentList.length > 0) {
                              elements.push(
                                <ul key={`list-${index}-split-before`} className="list-disc list-inside space-y-2 ml-6 mb-6">
                                  {currentList.map((item, i) => (
                                    <li key={i} className="mb-1.5">{item}</li>
                                  ))}
                                </ul>
                              );
                              currentList = [];
                              inList = false;
                              lastWasList = true;
                            }
                            
                            // Split by bullet (with or without spaces around it)
                            // Pattern: split on "•" with optional spaces before and after
                            const items = trimmed.split(/\s*•\s*/).filter(item => {
                              const cleaned = item.trim();
                              return cleaned && cleaned !== "•" && cleaned.length > 0;
                            });
                            
                            if (items.length > 1) {
                              // Multiple items in one line - create list immediately
                              elements.push(
                                <ul key={`list-${index}-split`} className="list-disc list-inside space-y-2 ml-6 mb-6">
                                  {items.map((item, i) => {
                                    const cleanItem = item.trim();
                                    return cleanItem ? (
                                      <li key={i} className="mb-1.5">{cleanItem}</li>
                                    ) : null;
                                  })}
                                </ul>
                              );
                              lastWasList = true;
                              return;
                            } else if (items.length === 1) {
                              // Single item after split - treat as list item
                              if (!inList) {
                                inList = true;
                              }
                              const cleanItem = items[0].trim();
                              if (cleanItem) {
                                currentList.push(cleanItem);
                              }
                              return;
                            }
                          }
                          
                          // Check if it's a bullet point (starts with • or -)
                          if (trimmed.startsWith("•") || trimmed.startsWith("-")) {
                            if (!inList) {
                              // Close previous list if exists and add spacing
                              if (currentList.length > 0) {
                                elements.push(
                                  <ul key={`list-${index}-before`} className="list-disc list-inside space-y-2 ml-6 mb-6">
                                    {currentList.map((item, i) => (
                                      <li key={i} className="mb-1.5">{item}</li>
                                    ))}
                                  </ul>
                                );
                                currentList = [];
                                lastWasList = true;
                              }
                              inList = true;
                            }
                            // Remove bullet and add to list
                            const cleanItem = trimmed.replace(/^[•\-]\s*/, "").trim();
                            if (cleanItem) {
                              currentList.push(cleanItem);
                            }
                            return;
                          }
                          
                          // Close list if we encounter non-bullet content
                          if (inList && currentList.length > 0) {
                            elements.push(
                              <ul key={`list-${index}-close`} className="list-disc list-inside space-y-2 ml-6 mb-6">
                                {currentList.map((item, i) => (
                                  <li key={i} className="mb-1.5">{item.replace(/^[•\-]\s*/, "").trim()}</li>
                                ))}
                              </ul>
                            );
                            currentList = [];
                            inList = false;
                            lastWasList = true;
                          }
                          
                          // Check if it's a section header (all caps or ends with colon)
                          const isAllCaps = trimmed === trimmed.toUpperCase() && trimmed.length > 3 && /[A-ZÁÉÍÓÚÇ]/.test(trimmed);
                          if (trimmed.endsWith(":") || isAllCaps) {
                            // Add spacing before section header if previous was list
                            if (lastWasList) {
                              elements.push(<div key={`spacer-${index}`} className="mb-2" />);
                              lastWasList = false;
                            }
                            elements.push(
                              <h3 key={index} className="font-semibold text-foreground text-lg mt-8 mb-4 first:mt-0">
                                {trimmed}
                              </h3>
                            );
                            return;
                          }
                          
                          // Regular paragraph - add spacing if previous was list
                          if (lastWasList) {
                            elements.push(<div key={`spacer-p-${index}`} className="mb-2" />);
                            lastWasList = false;
                          }
                          
                          elements.push(
                            <p key={index} className="mb-4 leading-relaxed">
                              {trimmed}
                            </p>
                          );
                        });
                        
                        // Close any remaining list
                        if (inList && currentList.length > 0) {
                          elements.push(
                            <ul key="list-final" className="list-disc list-inside space-y-2 ml-6 mb-6">
                              {currentList.map((item, i) => (
                                <li key={i} className="mb-1.5">{item.replace(/^[•\-]\s*/, "").trim()}</li>
                              ))}
                            </ul>
                          );
                        }
                        
                        return elements;
                      })()}
                    </div>
                  )}

                  {/* For video lessons - show description below video */}
                  {hasVideo && currentLesson.description && (
                    <div className="text-foreground leading-relaxed space-y-4">
                      {(() => {
                        const formatted = formatText(currentLesson.description);
                        const lines = formatted.split("\n");
                        const elements: JSX.Element[] = [];
                        let currentList: string[] = [];
                        let inList = false;
                        let lastWasList = false;
                        
                        lines.forEach((line, index) => {
                          const trimmed = line.trim();
                          if (!trimmed) {
                            // Empty line - close list if open
                            if (inList && currentList.length > 0) {
                              elements.push(
                                <ul key={`list-${index}`} className="list-disc list-inside space-y-2 ml-6 mb-4">
                                  {currentList.map((item, i) => (
                                    <li key={i} className="mb-1.5">{item}</li>
                                  ))}
                                </ul>
                              );
                              currentList = [];
                              inList = false;
                              lastWasList = true;
                            } else if (lastWasList) {
                              lastWasList = false;
                            }
                            return;
                          }
                          
                          // Check for timestamp pattern: "00:06 – texto" or "00:06 - texto"
                          const timestampMatch = trimmed.match(/^(\d{1,2}):(\d{2})\s*[–-]\s*(.+)$/);
                          if (timestampMatch) {
                            // Close list if open
                            if (inList && currentList.length > 0) {
                              elements.push(
                                <ul key={`list-${index}-before-ts`} className="list-disc list-inside space-y-2 ml-6 mb-4">
                                  {currentList.map((item, i) => (
                                    <li key={i} className="mb-1.5">{item}</li>
                                  ))}
                                </ul>
                              );
                              currentList = [];
                              inList = false;
                              lastWasList = true;
                            }
                            
                            // Render timestamp as clickable item
                            const [, minutes, seconds, text] = timestampMatch;
                            const totalSeconds = parseInt(minutes) * 60 + parseInt(seconds);
                            const timeDisplay = `${minutes}:${seconds}`;
                            
                            elements.push(
                              <button
                                key={index}
                                onClick={() => {
                                  if (videoRef.current) {
                                    videoRef.current.currentTime = totalSeconds;
                                    videoRef.current.play();
                                    setIsPlaying(true);
                                  }
                                }}
                                className="mb-3 flex items-start gap-3 w-full text-left hover:bg-muted p-2 rounded-md transition-colors group"
                              >
                                <span className="font-mono text-primary font-semibold text-sm mt-0.5 flex-shrink-0 group-hover:text-primary/80 transition-colors">
                                  {timeDisplay}
                                </span>
                                <span className="flex-1">{text}</span>
                              </button>
                            );
                            lastWasList = false;
                            return;
                          }
                          
                          // Check if line contains multiple items separated by bullet (•)
                          if (trimmed.includes("•") && (trimmed.match(/•/g) || []).length > 0) {
                            // Close current list if open
                            if (inList && currentList.length > 0) {
                              elements.push(
                                <ul key={`list-${index}-split-before`} className="list-disc list-inside space-y-2 ml-6 mb-4">
                                  {currentList.map((item, i) => (
                                    <li key={i} className="mb-1.5">{item}</li>
                                  ))}
                                </ul>
                              );
                              currentList = [];
                              inList = false;
                              lastWasList = true;
                            }
                            
                            // Split by bullet
                            const items = trimmed.split(/\s*•\s*/).filter(item => {
                              const cleaned = item.trim();
                              return cleaned && cleaned !== "•" && cleaned.length > 0;
                            });
                            
                            if (items.length > 1) {
                              elements.push(
                                <ul key={`list-${index}-split`} className="list-disc list-inside space-y-2 ml-6 mb-4">
                                  {items.map((item, i) => {
                                    const cleanItem = item.trim();
                                    return cleanItem ? (
                                      <li key={i} className="mb-1.5">{cleanItem}</li>
                                    ) : null;
                                  })}
                                </ul>
                              );
                              lastWasList = true;
                              return;
                            } else if (items.length === 1) {
                              if (!inList) {
                                inList = true;
                              }
                              const cleanItem = items[0].trim();
                              if (cleanItem) {
                                currentList.push(cleanItem);
                              }
                              return;
                            }
                          }
                          
                          // Check if it's a bullet point (starts with • or -)
                          if (trimmed.startsWith("•") || trimmed.startsWith("-")) {
                            if (!inList) {
                              if (currentList.length > 0) {
                                elements.push(
                                  <ul key={`list-${index}-before`} className="list-disc list-inside space-y-2 ml-6 mb-4">
                                    {currentList.map((item, i) => (
                                      <li key={i} className="mb-1.5">{item}</li>
                                    ))}
                                  </ul>
                                );
                                currentList = [];
                                lastWasList = true;
                              }
                              inList = true;
                            }
                            const cleanItem = trimmed.replace(/^[•\-]\s*/, "").trim();
                            if (cleanItem) {
                              currentList.push(cleanItem);
                            }
                            return;
                          }
                          
                          // Close list if we encounter non-bullet content
                          if (inList && currentList.length > 0) {
                            elements.push(
                              <ul key={`list-${index}-close`} className="list-disc list-inside space-y-2 ml-6 mb-4">
                                {currentList.map((item, i) => (
                                  <li key={i} className="mb-1.5">{item}</li>
                                ))}
                              </ul>
                            );
                            currentList = [];
                            inList = false;
                            lastWasList = true;
                          }
                          
                          // Check if it's a section header
                          const isAllCaps = trimmed === trimmed.toUpperCase() && trimmed.length > 3 && /[A-ZÁÉÍÓÚÇ]/.test(trimmed);
                          if (trimmed.endsWith(":") || isAllCaps) {
                            if (lastWasList) {
                              elements.push(<div key={`spacer-${index}`} className="mb-2" />);
                              lastWasList = false;
                            }
                            elements.push(
                              <h3 key={index} className="font-semibold text-foreground text-lg mt-6 mb-3 first:mt-0">
                                {trimmed}
                              </h3>
                            );
                            return;
                          }
                          
                          // Regular paragraph
                          if (lastWasList) {
                            elements.push(<div key={`spacer-p-${index}`} className="mb-2" />);
                            lastWasList = false;
                          }
                          
                          elements.push(
                            <p key={index} className="mb-4 leading-relaxed">
                              {trimmed}
                            </p>
                          );
                        });
                        
                        // Close any remaining list
                        if (inList && currentList.length > 0) {
                          elements.push(
                            <ul key="list-final" className="list-disc list-inside space-y-2 ml-6 mb-4">
                              {currentList.map((item, i) => (
                                <li key={i} className="mb-1.5">{item}</li>
                              ))}
                            </ul>
                          );
                        }
                        
                        return elements;
                      })()}
                    </div>
                  )}
                </div>
              )}
            </div>

            {/* Sidebar - Right Side */}
            <div className="lg:col-span-1 order-first lg:order-last">
              <div className="sticky top-20 lg:top-24 space-y-4 sm:space-y-6">
                {/* Progress Card */}
                <Card className="bg-card border-border">
                  <CardContent className="p-4 sm:p-6">
                    <h3 className="text-card-foreground font-semibold mb-3 sm:mb-4 text-sm sm:text-base">Meu Progresso - {courseData?.overallProgress || 0}%</h3>
                    <p className="text-muted-foreground text-xs sm:text-sm mb-3 sm:mb-4">
                      {completedLessons} de {totalLessons} aulas
                    </p>
                    {/* Circular Progress */}
                    <div className="relative w-20 h-20 sm:w-24 sm:h-24 mx-auto mb-3 sm:mb-4">
                      <svg className="w-20 h-20 sm:w-24 sm:h-24 transform -rotate-90" viewBox="0 0 100 100">
                        <circle
                          cx="50"
                          cy="50"
                          r="40"
                          stroke="hsl(var(--muted))"
                          strokeWidth="8"
                          fill="none"
                        />
                        <circle
                          cx="50"
                          cy="50"
                          r="40"
                          stroke="hsl(var(--primary))"
                          strokeWidth="8"
                          fill="none"
                          strokeDasharray={`${(courseData?.overallProgress || 0) * 2.51} 251`}
                          strokeLinecap="round"
                        />
                      </svg>
                      <div className="absolute inset-0 flex items-center justify-center">
                        <span className="text-card-foreground font-semibold text-base sm:text-lg">
                          {courseData?.overallProgress || 0}%
                        </span>
                      </div>
                    </div>
                  </CardContent>
                </Card>

                {/* Modules Timeline */}
                <Card className="bg-card border-border">
                  <CardContent className="p-4 sm:p-6">
                    <div className="space-y-4 sm:space-y-6">
                      {courseData?.modules.map((module, moduleIndex) => {
                        const moduleLessons = module.lessons;
                        const moduleCompleted = moduleLessons.every(l => l.completed);
                        const hasActiveLesson = moduleLessons.some(l => l.id === currentLessonId);
                        const isExpanded = expandedModuleId === module.id;

                        return (
                          <div key={module.id} className="relative">
                            {/* Timeline Circle */}
                            <div className="flex items-start gap-4">
                              <div className="relative flex flex-col items-center">
                                <div
                                  className={`w-4 h-4 rounded-full ${
                                    moduleCompleted
                                      ? "bg-primary"
                                      : hasActiveLesson
                                      ? "bg-primary"
                                      : "bg-muted"
                                  }`}
                                />
                                {moduleIndex < (courseData?.modules.length || 0) - 1 && (
                                  <div className="absolute top-4 left-1/2 transform -translate-x-1/2 w-0.5 h-16 bg-border" />
                                )}
                              </div>

                              <div className="flex-1 min-w-0">
                                {/* Module Header - Clickable to expand/collapse */}
                                <button
                                  onClick={() => {
                                    if (isExpanded) {
                                      setExpandedModuleId(null);
                                    } else {
                                      setExpandedModuleId(module.id);
                                    }
                                  }}
                                  className="w-full flex items-center justify-between mb-2 hover:opacity-80 transition-opacity"
                                >
                                  <div className="flex-1 text-left">
                                    <h4 className="text-card-foreground font-medium text-sm sm:text-base">{module.title}</h4>
                                    <p className="text-muted-foreground text-sm">
                                      {moduleLessons.length} {moduleLessons.length === 1 ? "aula" : "aulas"}
                                    </p>
                                  </div>
                                  <div className="flex-shrink-0 ml-2">
                                    {isExpanded ? (
                                      <ChevronDown className="w-5 h-5 text-muted-foreground" />
                                    ) : (
                                      <ChevronRight className="w-5 h-5 text-muted-foreground" />
                                    )}
                                  </div>
                                </button>

                                {/* Lessons List - Only show when expanded */}
                                {isExpanded && (
                                  <div className="space-y-2 mt-3">
                                    {moduleLessons.map((lesson) => {
                                      const isActive = lesson.id === currentLessonId;
                                      return (
                                        <div
                                          key={lesson.id}
                                          className={`w-full p-3 rounded-md transition-colors ${
                                            isActive
                                              ? "bg-primary/20 text-primary border border-primary/30"
                                              : "hover:bg-muted text-card-foreground"
                                          }`}
                                          data-testid={`lesson-${lesson.id}`}
                                        >
                                          <div className="flex items-start gap-3">
                                            {/* Clickable circle/checkmark to mark as completed */}
                                            <button
                                              onClick={(e) => {
                                                e.stopPropagation();
                                                if (!lesson.completed) {
                                                  markCompleteMutation.mutate(lesson.id);
                                                }
                                              }}
                                              className="mt-0.5 flex-shrink-0 hover:scale-110 transition-transform"
                                              title={lesson.completed ? "Aula concluída" : "Marcar como concluída"}
                                              disabled={markCompleteMutation.isPending}
                                            >
                                              {lesson.completed ? (
                                                <CheckCircle className="w-5 h-5 text-primary cursor-pointer" />
                                              ) : (
                                                <Circle className={`w-5 h-5 ${isActive ? "text-primary" : "text-muted-foreground"} cursor-pointer hover:text-primary transition-colors`} />
                                              )}
                                            </button>
                                            
                                            {/* Clickable lesson title */}
                                            <button
                                              onClick={() => setCurrentLessonId(lesson.id)}
                                              className="flex-1 min-w-0 text-left"
                                            >
                                              <p className={`text-sm line-clamp-2 ${isActive ? "font-medium" : ""}`}>
                                                {lesson.title}
                                              </p>
                                              {lesson.videoDuration && (
                                                <p className={`text-xs mt-1 ${isActive ? "text-primary/70" : "text-muted-foreground"}`}>
                                                  {Math.floor(lesson.videoDuration / 60)}:{String(lesson.videoDuration % 60).padStart(2, '0')} min
                                                </p>
                                              )}
                                            </button>
                                          </div>
                                        </div>
                                      );
                                    })}
                                  </div>
                                )}

                                {/* New Content Indicator */}
                                {moduleLessons.some(l => !l.completed) && (
                                  <div className="mt-2 flex items-center gap-2 text-muted-foreground text-xs">
                                    <Circle className="w-2 h-2" />
                                    <span>Novo conteúdo</span>
                                  </div>
                                )}
                              </div>
                            </div>
                          </div>
                        );
                      })}
                    </div>
                  </CardContent>
                </Card>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
