// Admin Vitrine - Gestão de vitrines de cursos com drag-and-drop
import { useEffect, useState } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useMutation, useQuery } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { DndContext, DragEndEvent, closestCenter } from "@dnd-kit/core";
import {
  SortableContext,
  arrayMove,
  useSortable,
  verticalListSortingStrategy,
} from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import {
  ArrowDown,
  ArrowUp,
  ArrowUpRight,
  ChevronDown,
  EllipsisVertical,
  EyeOff,
  Globe,
  GripVertical,
  PenLine,
  Plus,
  Trash2,
} from "lucide-react";
import type { Course, CourseShowcase } from "@shared/schema";
import { useLocation } from "wouter";
import { cn } from "@/lib/utils";

type CourseWithStats = Course & { lessonsCount?: number };

type ShowcaseWithCourses = CourseShowcase & { courses: CourseWithStats[] };

type CourseStatus = Course["status"];

function getStatusBadgeVariant(_status: CourseStatus) {
  return "outline" as const;
}

function getStatusLabel(status: CourseStatus) {
  switch (status) {
    case "published":
      return "Publicado";
    case "draft":
      return "Não listado";
    default:
      return "Arquivado";
  }
}

function getStatusClasses(status: CourseStatus) {
  switch (status) {
    case "published":
      return "bg-emerald-50 text-emerald-700 border-emerald-100";
    case "draft":
      return "bg-amber-50 text-amber-700 border-amber-100";
    default:
      return "bg-slate-100 text-slate-600 border-transparent";
  }
}

interface SortableCourseRowProps {
  course: CourseWithStats;
  onOpenCourse: () => void;
  onRemoveCourse: () => void;
  onChangeStatus: (status: CourseStatus) => void;
  statusPending: boolean;
  onRenameCourse: (title: string) => Promise<void>;
  renamePending: boolean;
}

function SortableCourseRow({
  course,
  onOpenCourse,
  onRemoveCourse,
  onChangeStatus,
  statusPending,
  onRenameCourse,
  renamePending,
}: SortableCourseRowProps) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({
    id: course.id,
  });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
  };

  const [isRenameDialogOpen, setIsRenameDialogOpen] = useState(false);
  const [renameTitle, setRenameTitle] = useState(course.title);

  useEffect(() => {
    if (!isRenameDialogOpen) {
      setRenameTitle(course.title);
    }
  }, [course.title, isRenameDialogOpen]);

  const handleStatusSelection = (nextStatus: CourseStatus) => {
    if (statusPending || course.status === nextStatus) {
      return;
    }
    onChangeStatus(nextStatus);
  };

  const handleRenameSubmit = async (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    const trimmed = renameTitle.trim();
    if (!trimmed || trimmed === course.title) {
      return;
    }
    try {
      await onRenameCourse(trimmed);
      setIsRenameDialogOpen(false);
    } catch {
      // handled by parent mutation
    }
  };

  return (
    <>
      <Dialog open={isRenameDialogOpen} onOpenChange={setIsRenameDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Renomear curso</DialogTitle>
            <DialogDescription>Defina um novo título para este curso.</DialogDescription>
          </DialogHeader>
          <form onSubmit={handleRenameSubmit} className="space-y-4">
            <div className="space-y-2">
              <label className="text-sm font-medium" htmlFor={`rename-course-${course.id}`}>
                Nome do curso
              </label>
              <Input
                id={`rename-course-${course.id}`}
                value={renameTitle}
                onChange={(event) => setRenameTitle(event.target.value)}
                autoFocus
              />
            </div>
            <DialogFooter>
              <Button type="button" variant="outline" onClick={() => setIsRenameDialogOpen(false)}>
                Cancelar
              </Button>
              <Button
                type="submit"
                disabled={
                  renamePending ||
                  !renameTitle.trim() ||
                  renameTitle.trim() === course.title
                }
              >
                {renamePending ? "Salvando..." : "Salvar"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      <div
        ref={setNodeRef}
        style={style}
        className={cn(
          "flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between rounded-2xl border border-muted-foreground/10 bg-card p-4 sm:p-5 transition",
          isDragging ? "ring-2 ring-primary/60 shadow-lg" : "shadow-sm",
        )}
      >
        <div className="flex flex-1 items-center gap-3 sm:gap-5 min-w-0">
          <button
            className="flex h-10 w-10 sm:h-12 sm:w-12 flex-shrink-0 items-center justify-center rounded-xl border border-muted/30 bg-muted/60 text-muted-foreground transition hover:bg-muted"
            type="button"
            aria-label={`Reordenar curso ${course.title}`}
            {...attributes}
            {...listeners}
          >
            <GripVertical className="h-3 w-3 sm:h-4 sm:w-4" />
          </button>

          <div className="flex items-center gap-3 sm:gap-4 min-w-0 flex-1">
            <div className="h-16 w-16 sm:h-20 sm:w-20 flex-shrink-0 overflow-hidden rounded-xl border border-muted/20 bg-muted">
              {course.thumbnail || course.thumbnailUrl ? (
                <img
                  src={
                    course.thumbnail
                      ? `data:${course.thumbnailMimeType};base64,${course.thumbnail}`
                      : course.thumbnailUrl || ""
                  }
                  alt={course.title}
                  className="h-full w-full object-cover"
                />
              ) : (
                <div className="flex h-full w-full items-center justify-center text-xs text-muted-foreground">
                  Sem capa
                </div>
              )}
            </div>
            <div className="min-w-0 flex-1">
              <p className="text-base sm:text-lg font-semibold leading-tight truncate">{course.title}</p>
              <p className="text-xs sm:text-sm text-muted-foreground truncate">
                {course.category || course.instructorName || "Categoria não informada"}
              </p>
            </div>
          </div>
        </div>

        <div className="flex items-center gap-2 sm:gap-3 flex-shrink-0">
          <div className="text-center">
            <p className="text-sm sm:text-lg font-semibold">{course.lessonsCount ?? 0}</p>
            <p className="text-[10px] sm:text-xs uppercase tracking-wide text-muted-foreground">aulas</p>
          </div>

          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button
                variant="ghost"
                size="sm"
                className={cn(
                  "flex items-center gap-1 sm:gap-2 rounded-full border px-2 sm:px-4 py-2 text-xs sm:text-sm font-medium shadow-none",
                  getStatusClasses(course.status),
                )}
                disabled={statusPending}
              >
                <span className="hidden sm:inline">{getStatusLabel(course.status)}</span>
                <span className="sm:hidden">{getStatusLabel(course.status).split(' ')[0]}</span>
                <ChevronDown className="h-3 w-3 opacity-70" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">
              <DropdownMenuItem
                onSelect={(event) => {
                  event.preventDefault();
                  handleStatusSelection("published");
                }}
                disabled={statusPending || course.status === "published"}
              >
                <Globe className="mr-2 h-4 w-4 text-emerald-600" />
                Publicado
              </DropdownMenuItem>
              <DropdownMenuItem
                onSelect={(event) => {
                  event.preventDefault();
                  handleStatusSelection("draft");
                }}
                disabled={statusPending || course.status === "draft"}
              >
                <EyeOff className="mr-2 h-4 w-4 text-amber-600" />
                Não listado
              </DropdownMenuItem>
              <DropdownMenuItem
                onSelect={(event) => {
                  event.preventDefault();
                  handleStatusSelection("archived");
                }}
                disabled={statusPending || course.status === "archived"}
              >
                <Trash2 className="mr-2 h-4 w-4 text-muted-foreground" />
                Arquivado
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>

          <Button
            variant="ghost"
            size="icon"
            onClick={onOpenCourse}
            title="Abrir curso"
            data-testid={`button-open-course-${course.id}`}
            className="h-8 w-8 sm:h-10 sm:w-10"
          >
            <ArrowUpRight className="h-4 w-4 sm:h-5 sm:w-5" />
          </Button>

          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="ghost" size="icon" className="h-8 w-8 sm:h-10 sm:w-10">
                <EllipsisVertical className="h-4 w-4 sm:h-5 sm:w-5" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end" className="w-40 sm:w-48 space-y-1">
              <DropdownMenuItem
                onSelect={(event) => {
                  event.preventDefault();
                  setIsRenameDialogOpen(true);
                }}
              >
                <PenLine className="mr-2 h-4 w-4" />
                Renomear
              </DropdownMenuItem>
              <DropdownMenuItem
                onSelect={(event) => {
                  event.preventDefault();
                  onRemoveCourse();
                }}
                className="text-destructive focus:text-destructive"
              >
                <Trash2 className="mr-2 h-4 w-4" />
                Excluir
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </div>
    </>
  );
}

interface ShowcaseCardProps {
  showcase: ShowcaseWithCourses;
  availableCourses: Course[];
  isAddCourseDialogOpen: boolean;
  onOpenCourseDialog: (open: boolean) => void;
  onAddCourse: (event: React.FormEvent<HTMLFormElement>) => void;
  onDeleteShowcase: () => void;
  onOpenCourse: (courseId: string) => void;
  onRemoveCourse: (courseId: string) => void;
  onDragEnd: (event: DragEndEvent) => void;
  selectedCourseId: string;
  onSelectCourse: (value: string) => void;
  addCoursePending: boolean;
  onCreateCourse: (data: { title: string; category: string; thumbnailUrl: string }) => Promise<void>;
  createCoursePending: boolean;
  onRenameShowcase: (name: string) => Promise<void>;
  renameShowcasePending: boolean;
  onShowcaseStatusChange: (status: CourseShowcase["status"]) => void;
  showcaseStatusPending: boolean;
  onCourseStatusChange: (courseId: string, status: CourseStatus) => void;
  onRenameCourse: (courseId: string, title: string) => Promise<void>;
  courseStatusPendingId: string | null;
  courseStatusMutationPending: boolean;
  courseRenamePendingId: string | null;
  courseRenameMutationPending: boolean;
  onMoveUp: () => void;
  onMoveDown: () => void;
  canMoveUp: boolean;
  canMoveDown: boolean;
  dragHandleProps?: {
    attributes: Record<string, unknown>;
    listeners: Record<string, unknown>;
  };
}

function ShowcaseCard({
  showcase,
  availableCourses,
  isAddCourseDialogOpen,
  onOpenCourseDialog,
  onAddCourse,
  onDeleteShowcase,
  onOpenCourse,
  onRemoveCourse,
  onDragEnd,
  selectedCourseId,
  onSelectCourse,
  addCoursePending,
  onCreateCourse,
  createCoursePending,
  onRenameShowcase,
  renameShowcasePending,
  onShowcaseStatusChange,
  showcaseStatusPending,
  onCourseStatusChange,
  onRenameCourse,
  courseStatusPendingId,
  courseStatusMutationPending,
  courseRenamePendingId,
  courseRenameMutationPending,
  onMoveUp,
  onMoveDown,
  canMoveUp,
  canMoveDown,
  dragHandleProps,
}: ShowcaseCardProps) {
  const [isCreateCourseDialogOpen, setIsCreateCourseDialogOpen] = useState(false);
  const [isRenameDialogOpen, setIsRenameDialogOpen] = useState(false);
  const [newCourseTitle, setNewCourseTitle] = useState("");
  const [newCourseCategory, setNewCourseCategory] = useState("");
  const [newCourseThumbnailUrl, setNewCourseThumbnailUrl] = useState("");
  const [renameValue, setRenameValue] = useState(showcase.name);

  useEffect(() => {
    if (!isRenameDialogOpen) {
      setRenameValue(showcase.name);
    }
  }, [showcase.name, isRenameDialogOpen]);

  const status = showcase.status ?? "unlisted";
  const isListed = status === "listed";
  const statusLabel = isListed ? "Publicado" : "Não listado";
  const StatusIcon = isListed ? Globe : EyeOff;
  const statusClasses = isListed
    ? "bg-emerald-50 text-emerald-700 border-emerald-100"
    : "bg-amber-50 text-amber-700 border-amber-100";

  const resetCreateCourseForm = () => {
    setNewCourseTitle("");
    setNewCourseCategory("");
    setNewCourseThumbnailUrl("");
  };

  const handleCreateCourseSubmit = async (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!newCourseTitle || !newCourseCategory || !newCourseThumbnailUrl) {
      return;
    }
    try {
      await onCreateCourse({
        title: newCourseTitle,
        category: newCourseCategory,
        thumbnailUrl: newCourseThumbnailUrl,
      });
      resetCreateCourseForm();
      setIsCreateCourseDialogOpen(false);
    } catch {
      // handled by mutation
    }
  };

  const handleRenameShowcaseSubmit = async (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    const trimmed = renameValue.trim();
    if (!trimmed || trimmed === showcase.name) return;
    try {
      await onRenameShowcase(trimmed);
      setIsRenameDialogOpen(false);
    } catch {
      // handled by mutation
    }
  };

  const handleShowcaseStatusSelect = (nextStatus: CourseShowcase["status"]) => {
    if (showcaseStatusPending || status === nextStatus) return;
    onShowcaseStatusChange(nextStatus);
  };

  const openCreateCourseDialog = () => {
    resetCreateCourseForm();
    setIsCreateCourseDialogOpen(true);
  };

  return (
    <>
      <Dialog open={isAddCourseDialogOpen} onOpenChange={onOpenCourseDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Adicionar curso à vitrine</DialogTitle>
            <DialogDescription>
              Escolha um curso existente para adicionar a &quot;{showcase.name}&quot;.
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={onAddCourse} className="space-y-4">
            <div className="space-y-2">
              <label className="text-sm font-medium">Curso</label>
              <Select value={selectedCourseId} onValueChange={onSelectCourse}>
                <SelectTrigger data-testid="select-course-to-add">
                  <SelectValue placeholder="Selecione um curso" />
                </SelectTrigger>
                <SelectContent>
                  {availableCourses.length ? (
                    availableCourses.map((course) => (
                      <SelectItem key={course.id} value={course.id}>
                        {course.title}
                      </SelectItem>
                    ))
                  ) : (
                    <div className="p-2 text-sm text-muted-foreground">Nenhum curso disponível</div>
                  )}
                </SelectContent>
              </Select>
            </div>
            <DialogFooter>
              <Button type="button" variant="outline" onClick={() => onOpenCourseDialog(false)}>
                Cancelar
              </Button>
              <Button type="submit" disabled={!selectedCourseId || addCoursePending}>
                {addCoursePending ? "Adicionando..." : "Adicionar"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={isCreateCourseDialogOpen} onOpenChange={setIsCreateCourseDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Criar novo curso</DialogTitle>
            <DialogDescription>
              Preencha as informações básicas do curso. Ele será adicionado a &quot;{showcase.name}&quot;.
            </DialogDescription>
          </DialogHeader>
          <form onSubmit={handleCreateCourseSubmit} className="space-y-4">
            <div className="space-y-2">
              <label className="text-sm font-medium" htmlFor={`new-course-title-${showcase.id}`}>
                Nome do curso
              </label>
              <Input
                id={`new-course-title-${showcase.id}`}
                value={newCourseTitle}
                onChange={(event) => setNewCourseTitle(event.target.value)}
                placeholder="Ex.: Introdução ao Marketing Digital"
                required
              />
            </div>
            <div className="space-y-2">
              <label className="text-sm font-medium" htmlFor={`new-course-category-${showcase.id}`}>
                Categoria do curso
              </label>
              <Input
                id={`new-course-category-${showcase.id}`}
                value={newCourseCategory}
                onChange={(event) => setNewCourseCategory(event.target.value)}
                placeholder="Ex.: Marketing"
                required
              />
            </div>
            <div className="space-y-2">
              <label className="text-sm font-medium" htmlFor={`new-course-thumbnail-${showcase.id}`}>
                Thumbnail (URL)
              </label>
              <Input
                id={`new-course-thumbnail-${showcase.id}`}
                value={newCourseThumbnailUrl}
                onChange={(event) => setNewCourseThumbnailUrl(event.target.value)}
                placeholder="https://exemplo.com/thumbnail.jpg"
                type="url"
                required
              />
            </div>
            <DialogFooter>
              <Button type="button" variant="outline" onClick={() => setIsCreateCourseDialogOpen(false)}>
                Cancelar
              </Button>
              <Button
                type="submit"
                disabled={
                  createCoursePending ||
                  !newCourseTitle ||
                  !newCourseCategory ||
                  !newCourseThumbnailUrl
                }
              >
                {createCoursePending ? "Criando..." : "Criar curso"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      <Dialog open={isRenameDialogOpen} onOpenChange={setIsRenameDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Renomear vitrine</DialogTitle>
            <DialogDescription>Defina um novo nome para esta vitrine.</DialogDescription>
          </DialogHeader>
          <form onSubmit={handleRenameShowcaseSubmit} className="space-y-4">
            <div className="space-y-2">
              <label className="text-sm font-medium" htmlFor={`rename-showcase-${showcase.id}`}>
                Nome da vitrine
              </label>
              <Input
                id={`rename-showcase-${showcase.id}`}
                value={renameValue}
                onChange={(event) => setRenameValue(event.target.value)}
                autoFocus
              />
            </div>
            <DialogFooter>
              <Button type="button" variant="outline" onClick={() => setIsRenameDialogOpen(false)}>
                Cancelar
              </Button>
              <Button
                type="submit"
                disabled={
                  renameShowcasePending ||
                  !renameValue.trim() ||
                  renameValue.trim() === showcase.name
                }
              >
                {renameShowcasePending ? "Renomeando..." : "Salvar"}
              </Button>
            </DialogFooter>
          </form>
        </DialogContent>
      </Dialog>

      <div className="space-y-5 rounded-3xl border border-muted/20 bg-background p-4 sm:p-6 shadow-md">
        <div className="flex flex-col gap-4 sm:flex-row sm:flex-wrap sm:items-center sm:justify-between">
          <div className="flex items-center gap-2 sm:gap-4">
            <div className="flex items-center gap-1 sm:gap-2 text-muted-foreground">
              <button
                type="button"
                className="flex h-8 w-8 sm:h-10 sm:w-10 items-center justify-center rounded-full border border-muted/30 bg-muted/50"
                aria-label="Arrastar vitrine"
                {...(dragHandleProps?.attributes ?? {})}
                {...(dragHandleProps?.listeners ?? {})}
              >
                <GripVertical className="h-3 w-3 sm:h-4 sm:w-4" />
              </button>
              <div className="flex flex-col items-center gap-1">
                <button
                  type="button"
                  onClick={onMoveUp}
                  disabled={!canMoveUp}
                  className="flex h-5 w-5 sm:h-6 sm:w-6 items-center justify-center rounded border border-muted/30 disabled:opacity-30"
                  aria-label="Mover vitrine para cima"
                >
                  <ArrowUp className="h-2.5 w-2.5 sm:h-3 sm:w-3" />
                </button>
                <button
                  type="button"
                  onClick={onMoveDown}
                  disabled={!canMoveDown}
                  className="flex h-5 w-5 sm:h-6 sm:w-6 items-center justify-center rounded border border-muted/30 disabled:opacity-30"
                  aria-label="Mover vitrine para baixo"
                >
                  <ArrowDown className="h-2.5 w-2.5 sm:h-3 sm:w-3" />
                </button>
              </div>
            </div>
            <div>
              <h2 className="text-xl sm:text-2xl font-semibold">{showcase.name}</h2>
              <div className="mt-1 h-[1px] w-24 sm:w-32 bg-muted" />
            </div>
          </div>

          <div className="flex flex-wrap items-center gap-2">
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button
                  variant="ghost"
                  size="sm"
                  className={cn(
                    "flex items-center gap-1 sm:gap-2 rounded-full border px-2 sm:px-4 py-2 text-xs sm:text-sm font-medium shadow-none",
                    statusClasses,
                  )}
                  disabled={showcaseStatusPending}
                >
                  <StatusIcon className="h-3 w-3 sm:h-4 sm:w-4" />
                  <span className="hidden sm:inline">{statusLabel}</span>
                  <span className="sm:hidden">{statusLabel.split(' ')[0]}</span>
                  <ChevronDown className="h-3 w-3 opacity-70" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end" className="w-48">
                <DropdownMenuItem
                  onSelect={(event) => {
                    event.preventDefault();
                    handleShowcaseStatusSelect("listed");
                  }}
                  disabled={showcaseStatusPending || status === "listed"}
                  className="flex items-center gap-2"
                >
                  <Globe className="h-4 w-4 text-emerald-600" />
                  Publicado
                </DropdownMenuItem>
                <DropdownMenuItem
                  onSelect={(event) => {
                    event.preventDefault();
                    handleShowcaseStatusSelect("unlisted");
                  }}
                  disabled={showcaseStatusPending || status === "unlisted"}
                  className="flex items-center gap-2"
                >
                  <EyeOff className="h-4 w-4 text-amber-600" />
                  Não listado
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>

            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="outline" size="icon" className="h-8 w-8 sm:h-10 sm:w-10">
                  <EllipsisVertical className="h-3 w-3 sm:h-4 sm:w-4" />
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end" className="w-40 sm:w-48 space-y-1">
                <DropdownMenuItem
                  onSelect={(event) => {
                    event.preventDefault();
                    setIsRenameDialogOpen(true);
                  }}
                >
                  Renomear vitrine
                </DropdownMenuItem>
                <DropdownMenuItem
                  onSelect={(event) => {
                    event.preventDefault();
                    onOpenCourseDialog(true);
                  }}
                >
                  Adicionar curso existente
                </DropdownMenuItem>
                <DropdownMenuItem
                  onSelect={(event) => {
                    event.preventDefault();
                    openCreateCourseDialog();
                  }}
                >
                  Criar novo curso
                </DropdownMenuItem>
                <DropdownMenuItem onClick={onDeleteShowcase} className="text-destructive focus:text-destructive">
                  <Trash2 className="mr-2 h-4 w-4" />
                  Remover vitrine
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
        </div>

        <div className="space-y-3">
          {showcase.courses.length ? (
            <DndContext collisionDetection={closestCenter} onDragEnd={onDragEnd}>
              <SortableContext
                items={showcase.courses.map((course) => course.id)}
                strategy={verticalListSortingStrategy}
              >
                <div className="space-y-3">
                  {showcase.courses.map((course) => (
                    <SortableCourseRow
                      key={course.id}
                      course={course}
                      onOpenCourse={() => onOpenCourse(course.id)}
                      onRemoveCourse={() => onRemoveCourse(course.id)}
                      onChangeStatus={(status) => onCourseStatusChange(course.id, status)}
                      statusPending={
                        courseStatusMutationPending && courseStatusPendingId === course.id
                      }
                      onRenameCourse={(title) => onRenameCourse(course.id, title)}
                      renamePending={
                        courseRenameMutationPending && courseRenamePendingId === course.id
                      }
                    />
                  ))}
                </div>
              </SortableContext>
            </DndContext>
          ) : (
            <div className="rounded-lg border border-dashed p-6 text-center text-sm text-muted-foreground">
              Nenhum curso nesta vitrine. Utilize o menu para criar ou adicionar cursos.
            </div>
          )}
        </div>
      </div>
    </>
  );
}

interface SortableShowcaseProps extends ShowcaseCardProps {
  showcase: ShowcaseWithCourses;
}

function SortableShowcaseItem(props: SortableShowcaseProps) {
  const { showcase } = props;
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({
    id: showcase.id,
  });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
  };

  return (
    <div
      ref={setNodeRef}
      style={style}
      className={cn("transition", isDragging ? "ring-2 ring-primary/60 rounded-3xl shadow-lg" : "")}
    >
      <ShowcaseCard
        {...props}
        dragHandleProps={{ attributes, listeners }}
      />
    </div>
  );
}

export default function AdminVitrine() {
  const { isAuthenticated, isLoading } = useAuth();
  const { toast } = useToast();
  const [, setLocation] = useLocation();

  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [newShowcaseName, setNewShowcaseName] = useState("");
  const [activeShowcaseId, setActiveShowcaseId] = useState<string | null>(null);
  const [selectedCourseId, setSelectedCourseId] = useState<string>("");
  const [localShowcases, setLocalShowcases] = useState<ShowcaseWithCourses[]>([]);
  const [showcaseStatusUpdatingId, setShowcaseStatusUpdatingId] = useState<string | null>(null);
  const [renamingShowcaseId, setRenamingShowcaseId] = useState<string | null>(null);
  const [creatingCourseForShowcaseId, setCreatingCourseForShowcaseId] = useState<string | null>(null);
  const [courseStatusPendingId, setCourseStatusPendingId] = useState<string | null>(null);
  const [courseRenamePendingId, setCourseRenamePendingId] = useState<string | null>(null);

  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Sessão expirada",
        description: "Você foi desconectado. Redirecionando para o login...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: showcasesData, isLoading: showcasesLoading } = useQuery<ShowcaseWithCourses[]>({
    queryKey: ["/api/admin/vitrines"],
    enabled: isAuthenticated,
  });

  const { data: coursesData, isLoading: coursesLoading } = useQuery<Course[]>({
    queryKey: ["/api/admin/courses"],
    enabled: isAuthenticated,
  });

  useEffect(() => {
    if (showcasesData) {
      setLocalShowcases(showcasesData);
    }
  }, [showcasesData]);

  const updateCourseInLocalState = (courseId: string, updater: (course: Course) => Course) => {
    setLocalShowcases((prev) =>
      prev.map((showcase) => ({
        ...showcase,
        courses: showcase.courses.map((course) =>
          course.id === courseId ? updater(course) : course,
        ),
      })),
    );
  };

  const createShowcaseMutation = useMutation({
    mutationFn: async (payload: { name: string }) => {
      const response = await apiRequest("POST", "/api/admin/vitrines", payload);
      return await response.json();
    },
    onSuccess: () => {
      toast({ title: "Vitrine criada com sucesso!" });
      setIsCreateDialogOpen(false);
      setNewShowcaseName("");
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onError: () => {
      toast({ title: "Erro ao criar vitrine", variant: "destructive" });
    },
  });

  const deleteShowcaseMutation = useMutation({
    mutationFn: async (showcaseId: string) => {
      return await apiRequest("DELETE", `/api/admin/vitrines/${showcaseId}`);
    },
    onSuccess: () => {
      toast({ title: "Vitrine removida" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onError: () => {
      toast({ title: "Erro ao remover vitrine", variant: "destructive" });
    },
  });

  const addCourseMutation = useMutation({
    mutationFn: async ({ showcaseId, courseId }: { showcaseId: string; courseId: string }) => {
      return await apiRequest("POST", `/api/admin/vitrines/${showcaseId}/courses`, { courseId });
    },
    onSuccess: () => {
      toast({ title: "Curso adicionado à vitrine!" });
      setActiveShowcaseId(null);
      setSelectedCourseId("");
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onError: () => {
      toast({ title: "Erro ao adicionar curso", variant: "destructive" });
    },
  });

  const removeCourseMutation = useMutation({
    mutationFn: async ({ showcaseId, courseId }: { showcaseId: string; courseId: string }) => {
      return await apiRequest("DELETE", `/api/admin/vitrines/${showcaseId}/courses/${courseId}`);
    },
    onSuccess: () => {
      toast({ title: "Curso removido da vitrine" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onError: () => {
      toast({ title: "Erro ao remover curso", variant: "destructive" });
    },
  });

  const reorderCoursesMutation = useMutation({
    mutationFn: async ({ showcaseId, courseIds }: { showcaseId: string; courseIds: string[] }) => {
      return await apiRequest("PATCH", `/api/admin/vitrines/${showcaseId}/courses/order`, {
        courseIds,
      });
    },
    onError: () => {
      toast({ title: "Erro ao reordenar cursos", variant: "destructive" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
  });

  const reorderShowcasesMutation = useMutation({
    mutationFn: async (showcaseIds: string[]) => {
      return await apiRequest("PATCH", "/api/admin/vitrines/order", { showcaseIds });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onError: () => {
      toast({ title: "Erro ao reordenar vitrines", variant: "destructive" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
  });

  const updateShowcaseStatusMutation = useMutation({
    mutationFn: async ({ showcaseId, status }: { showcaseId: string; status: CourseShowcase["status"] }) => {
      return await apiRequest("PATCH", `/api/admin/vitrines/${showcaseId}/status`, { status });
    },
    onMutate: ({ showcaseId, status }) => {
      setShowcaseStatusUpdatingId(showcaseId);
      setLocalShowcases((prev) =>
        prev.map((item) => (item.id === showcaseId ? { ...item, status } : item)),
      );
    },
    onSuccess: () => {
      toast({ title: "Status da vitrine atualizado!" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onError: () => {
      toast({ title: "Erro ao atualizar status da vitrine", variant: "destructive" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onSettled: () => {
      setShowcaseStatusUpdatingId(null);
    },
  });

  const createCourseForShowcaseMutation = useMutation({
    mutationFn: async ({
      showcaseId,
      payload,
    }: {
      showcaseId: string;
      payload: { title: string; category: string; thumbnailUrl: string };
    }) => {
      const response = await apiRequest("POST", `/api/admin/vitrines/${showcaseId}/courses/simple`, payload);
      return await response.json();
    },
    onMutate: ({ showcaseId }) => {
      setCreatingCourseForShowcaseId(showcaseId);
    },
    onSuccess: (result, { showcaseId }) => {
      const createdCourse = result?.course as Course | undefined;
      if (createdCourse) {
        setLocalShowcases((prev) =>
          prev.map((item) =>
            item.id === showcaseId ? { ...item, courses: [...item.courses, createdCourse] } : item,
          ),
        );
      }
      toast({ title: "Curso criado e adicionado à vitrine!" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses"] });
    },
    onError: () => {
      toast({ title: "Erro ao criar curso", variant: "destructive" });
    },
    onSettled: () => {
      setCreatingCourseForShowcaseId(null);
    },
  });

  const renameShowcaseMutation = useMutation({
    mutationFn: async ({ showcaseId, name }: { showcaseId: string; name: string }) => {
      return await apiRequest("PATCH", `/api/admin/vitrines/${showcaseId}/basic`, { name });
    },
    onMutate: ({ showcaseId, name }) => {
      setRenamingShowcaseId(showcaseId);
      setLocalShowcases((prev) =>
        prev.map((item) => (item.id === showcaseId ? { ...item, name } : item)),
      );
    },
    onSuccess: () => {
      toast({ title: "Vitrine atualizada!" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onError: () => {
      toast({ title: "Erro ao renomear vitrine", variant: "destructive" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onSettled: () => {
      setRenamingShowcaseId(null);
    },
  });

  const updateCourseStatusMutation = useMutation({
    mutationFn: async ({ courseId, status }: { courseId: string; status: CourseStatus }) => {
      return await apiRequest("PATCH", `/api/admin/courses/${courseId}/basic`, { status });
    },
    onMutate: ({ courseId, status }) => {
      setCourseStatusPendingId(courseId);
      updateCourseInLocalState(courseId, (course) => ({ ...course, status }));
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses"] });
    },
    onError: () => {
      toast({ title: "Erro ao atualizar status do curso", variant: "destructive" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onSettled: () => {
      setCourseStatusPendingId(null);
    },
  });

  const renameCourseMutation = useMutation({
    mutationFn: async ({ courseId, title }: { courseId: string; title: string }) => {
      return await apiRequest("PATCH", `/api/admin/courses/${courseId}/basic`, { title });
    },
    onMutate: ({ courseId, title }) => {
      setCourseRenamePendingId(courseId);
      updateCourseInLocalState(courseId, (course) => ({ ...course, title }));
    },
    onSuccess: () => {
      toast({ title: "Curso atualizado!" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses"] });
    },
    onError: () => {
      toast({ title: "Erro ao renomear curso", variant: "destructive" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/vitrines"] });
    },
    onSettled: () => {
      setCourseRenamePendingId(null);
    },
  });

  const handleCreateShowcase = (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    const trimmedName = newShowcaseName.trim();
    if (!trimmedName) {
      toast({ title: "Informe um nome para a vitrine", variant: "destructive" });
      return;
    }
    createShowcaseMutation.mutate({ name: trimmedName });
  };

  const handleDeleteShowcase = (showcaseId: string) => {
    if (deleteShowcaseMutation.isPending) return;
    if (confirm("Tem certeza de que deseja excluir esta vitrine?")) {
      deleteShowcaseMutation.mutate(showcaseId);
    }
  };

  const handleAddCourse = (event: React.FormEvent<HTMLFormElement>, showcaseId: string) => {
    event.preventDefault();
    if (!selectedCourseId) {
      toast({ title: "Selecione um curso", variant: "destructive" });
      return;
    }
    addCourseMutation.mutate({ showcaseId, courseId: selectedCourseId });
  };

  const handleRemoveCourse = (showcaseId: string, courseId: string) => {
    if (removeCourseMutation.isPending) return;
    setLocalShowcases((prev) =>
      prev.map((showcase) =>
        showcase.id === showcaseId
          ? { ...showcase, courses: showcase.courses.filter((course) => course.id !== courseId) }
          : showcase,
      ),
    );
    removeCourseMutation.mutate({ showcaseId, courseId });
  };

  const handleCourseDragEnd = (showcaseId: string) => (event: DragEndEvent) => {
    const { active, over } = event;
    if (!over || active.id === over.id) return;

    let reorderedCourses: Course[] | null = null;

    setLocalShowcases((prev) =>
      prev.map((showcase) => {
        if (showcase.id !== showcaseId) return showcase;

        const oldIndex = showcase.courses.findIndex((course) => course.id === active.id);
        const newIndex = showcase.courses.findIndex((course) => course.id === over.id);

        if (oldIndex === -1 || newIndex === -1) return showcase;

        reorderedCourses = arrayMove(showcase.courses, oldIndex, newIndex);
        return {
          ...showcase,
          courses: reorderedCourses,
        };
      }),
    );

    if (reorderedCourses) {
      reorderCoursesMutation.mutate({
        showcaseId,
        courseIds: reorderedCourses.map((course) => course.id),
      });
    }
  };

  const handleShowcaseDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;
    if (!over || active.id === over.id) return;

    let reordered: ShowcaseWithCourses[] | null = null;

    setLocalShowcases((prev) => {
      const oldIndex = prev.findIndex((item) => item.id === active.id);
      const newIndex = prev.findIndex((item) => item.id === over.id);
      if (oldIndex === -1 || newIndex === -1) {
        return prev;
      }
      reordered = arrayMove(prev, oldIndex, newIndex);
      return reordered;
    });

    if (reordered) {
      reorderShowcasesMutation.mutate(reordered.map((showcase) => showcase.id));
    }
  };

  const moveShowcase = (showcaseId: string, direction: "up" | "down") => {
    setLocalShowcases((prev) => {
      const index = prev.findIndex((item) => item.id === showcaseId);
      if (index === -1) return prev;
      const targetIndex = direction === "up" ? index - 1 : index + 1;
      if (targetIndex < 0 || targetIndex >= prev.length) return prev;
      const reordered = arrayMove(prev, index, targetIndex);
      reorderShowcasesMutation.mutate(reordered.map((showcase) => showcase.id));
      return reordered;
    });
  };

  const handleShowcaseStatusChange = (showcaseId: string, status: CourseShowcase["status"]) => {
    updateShowcaseStatusMutation.mutate({ showcaseId, status });
  };

  const handleRenameShowcase = (showcaseId: string, name: string) => {
    return renameShowcaseMutation.mutateAsync({ showcaseId, name });
  };

  const handleCourseStatusChange = (courseId: string, status: CourseStatus) => {
    updateCourseStatusMutation.mutate({ courseId, status });
  };

  const handleRenameCourse = (courseId: string, title: string) => {
    return renameCourseMutation.mutateAsync({ courseId, title });
  };

  const openAddCourseDialog = (showcaseId: string, defaultCourseId?: string) => {
    setActiveShowcaseId(showcaseId);
    setSelectedCourseId(defaultCourseId ?? "");
  };

  const handleAddCourseDialogChange = (showcaseId: string, open: boolean) => {
    if (open) {
      const showcase = localShowcases.find((sc) => sc.id === showcaseId);
      const available = coursesData?.filter(
        (course) => !showcase?.courses.some((c) => c.id === course.id),
      );
      openAddCourseDialog(showcaseId, available && available.length > 0 ? available[0].id : "");
    } else if (activeShowcaseId === showcaseId) {
      setActiveShowcaseId(null);
      setSelectedCourseId("");
    }
  };

  const isLoadingPage = showcasesLoading || coursesLoading;

  if (isLoadingPage) {
    return <div className="flex min-h-screen items-center justify-center">Carregando...</div>;
  }

  return (
    <div className="space-y-4 sm:space-y-6 lg:space-y-8">
      <div className="flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div className="flex-1">
          <h1 className="mb-2 text-2xl sm:text-3xl lg:text-4xl font-serif font-semibold">Vitrines</h1>
          <p className="text-sm sm:text-base text-muted-foreground">
            Organize os cursos em vitrines, arraste para reordenar e controle o status de visibilidade.
          </p>
        </div>

        <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
          <DialogTrigger asChild>
            <Button data-testid="button-create-showcase" className="w-full sm:w-auto">
              <Plus className="mr-2 h-4 w-4" />
              Nova Vitrine
            </Button>
          </DialogTrigger>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Criar nova vitrine</DialogTitle>
              <DialogDescription>Escolha um nome identificador para sua vitrine.</DialogDescription>
            </DialogHeader>
            <form onSubmit={handleCreateShowcase} className="space-y-4">
              <div className="space-y-2">
                <label htmlFor="showcase-name" className="text-sm font-medium">
                  Nome da vitrine
                </label>
                <Input
                  id="showcase-name"
                  value={newShowcaseName}
                  onChange={(event) => setNewShowcaseName(event.target.value)}
                  placeholder="Ex.: Acesso Rápido: Meus Cursos"
                  autoFocus
                  required
                  data-testid="input-showcase-name"
                />
              </div>
              <DialogFooter>
                <Button type="button" variant="outline" onClick={() => setIsCreateDialogOpen(false)}>
                  Cancelar
                </Button>
                <Button type="submit" disabled={createShowcaseMutation.isPending}>
                  {createShowcaseMutation.isPending ? "Salvando..." : "Salvar"}
                </Button>
              </DialogFooter>
            </form>
          </DialogContent>
        </Dialog>
      </div>

      <Card>
        <CardContent className="space-y-4 sm:space-y-6 py-4 sm:py-6">
          {!localShowcases.length ? (
            <div className="space-y-2 py-12 text-center text-muted-foreground">
              <p className="text-lg font-medium">Nenhuma vitrine cadastrada</p>
              <p className="text-sm">Use o botão &quot;Nova Vitrine&quot; para criar a primeira.</p>
            </div>
          ) : (
            <DndContext collisionDetection={closestCenter} onDragEnd={handleShowcaseDragEnd}>
              <SortableContext
                items={localShowcases.map((showcase) => showcase.id)}
                strategy={verticalListSortingStrategy}
              >
                <div className="space-y-6">
                  {localShowcases.map((showcase, index) => {
                    const availableCourses = coursesData?.filter(
                      (course) => !showcase.courses.some((selected) => selected.id === course.id),
                    );
                    const showcaseStatusPending =
                      updateShowcaseStatusMutation.isPending &&
                      showcaseStatusUpdatingId === showcase.id;
                    const renameShowcasePending =
                      renameShowcaseMutation.isPending &&
                      renamingShowcaseId === showcase.id;

                    return (
                      <SortableShowcaseItem
                        key={showcase.id}
                        showcase={showcase}
                        availableCourses={availableCourses ?? []}
                        isAddCourseDialogOpen={activeShowcaseId === showcase.id}
                        onOpenCourseDialog={(open) => handleAddCourseDialogChange(showcase.id, open)}
                        onAddCourse={(event) => handleAddCourse(event, showcase.id)}
                        onDeleteShowcase={() => handleDeleteShowcase(showcase.id)}
                        onOpenCourse={(courseId) => setLocation(`/admin/courses/${courseId}/content`)}
                        onRemoveCourse={(courseId) => handleRemoveCourse(showcase.id, courseId)}
                        onDragEnd={handleCourseDragEnd(showcase.id)}
                        selectedCourseId={selectedCourseId}
                        onSelectCourse={setSelectedCourseId}
                        addCoursePending={addCourseMutation.isPending}
                        onCreateCourse={(data) =>
                          createCourseForShowcaseMutation.mutateAsync({ showcaseId: showcase.id, payload: data })
                        }
                        createCoursePending={
                          createCourseForShowcaseMutation.isPending &&
                          creatingCourseForShowcaseId === showcase.id
                        }
                        onRenameShowcase={(name) => handleRenameShowcase(showcase.id, name)}
                        renameShowcasePending={renameShowcasePending}
                        onShowcaseStatusChange={(status) => handleShowcaseStatusChange(showcase.id, status)}
                        showcaseStatusPending={showcaseStatusPending}
                        onCourseStatusChange={handleCourseStatusChange}
                        onRenameCourse={handleRenameCourse}
                        courseStatusPendingId={courseStatusPendingId}
                        courseStatusMutationPending={updateCourseStatusMutation.isPending}
                        courseRenamePendingId={courseRenamePendingId}
                        courseRenameMutationPending={renameCourseMutation.isPending}
                        onMoveUp={() => moveShowcase(showcase.id, "up")}
                        onMoveDown={() => moveShowcase(showcase.id, "down")}
                        canMoveUp={index > 0}
                        canMoveDown={index < localShowcases.length - 1}
                      />
                    );
                  })}
                </div>
              </SortableContext>
            </DndContext>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
