// Admin Users - User management with XLSX export
import { useState, useEffect } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useQuery, useMutation } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Checkbox } from "@/components/ui/checkbox";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger, DialogFooter } from "@/components/ui/dialog";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Download, Search, UserPlus, Shield, GraduationCap, BookOpen } from "lucide-react";
import type { User, Course } from "@shared/schema";

export default function AdminUsers() {
  const { isAuthenticated, isLoading } = useAuth();
  const { toast } = useToast();
  const [searchTerm, setSearchTerm] = useState("");
  const [roleFilter, setRoleFilter] = useState<string>("all");
  const [selectedUser, setSelectedUser] = useState<User | null>(null);
  const [selectedCourses, setSelectedCourses] = useState<string[]>([]);
  const [isGrantAccessDialogOpen, setIsGrantAccessDialogOpen] = useState(false);

  // Redirect if not authenticated
  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: users, isLoading: usersLoading } = useQuery<User[]>({
    queryKey: ["/api/admin/users"],
    enabled: isAuthenticated,
  });

  const { data: courses } = useQuery<Course[]>({
    queryKey: ["/api/admin/courses"],
    enabled: isAuthenticated,
  });

  // Fetch user's current course access when dialog opens
  const { data: userCoursesIds } = useQuery<string[]>({
    queryKey: ["/api/admin/users", selectedUser?.id, "courses"],
    queryFn: async () => {
      if (!selectedUser) return [];
      const res = await fetch(`/api/admin/users/${selectedUser.id}/courses`, {
        credentials: "include",
      });
      if (!res.ok) throw new Error("Failed to fetch user courses");
      return res.json();
    },
    enabled: !!selectedUser && isGrantAccessDialogOpen,
  });

  // Update selected courses when user courses load
  useEffect(() => {
    if (userCoursesIds && isGrantAccessDialogOpen) {
      setSelectedCourses(userCoursesIds);
    }
  }, [userCoursesIds, isGrantAccessDialogOpen]);

  const grantAccessMutation = useMutation({
    mutationFn: async () => {
      if (!selectedUser || !userCoursesIds) return;
      
      const currentCourses = new Set(userCoursesIds);
      const newCourses = new Set(selectedCourses);
      
      // Find courses to add (in new but not in current)
      const toAdd = selectedCourses.filter(id => !currentCourses.has(id));
      
      // Find courses to remove (in current but not in new)
      const toRemove = userCoursesIds.filter(id => !newCourses.has(id));
      
      // Grant access to new courses
      const grantPromises = toAdd.map(courseId =>
        apiRequest("POST", "/api/admin/user-access", {
          userId: selectedUser.id,
          courseId,
        })
      );
      
      // Revoke access from removed courses
      const revokePromises = toRemove.map(async (courseId) => {
        const res = await fetch(`/api/admin/user-access/${selectedUser.id}/${courseId}`, {
          method: "DELETE",
          credentials: "include",
        });
        if (!res.ok) throw new Error("Failed to revoke access");
      });
      
      await Promise.all([...grantPromises, ...revokePromises]);
    },
    onSuccess: () => {
      toast({ title: "Permissões de cursos atualizadas!" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/users", selectedUser?.id, "courses"] });
      setIsGrantAccessDialogOpen(false);
      setSelectedUser(null);
      setSelectedCourses([]);
    },
    onError: () => {
      toast({ title: "Erro ao atualizar permissões", variant: "destructive" });
    },
  });

  const exportMutation = useMutation({
    mutationFn: async () => {
      const response = await fetch("/api/admin/users/export", {
        method: "GET",
      });
      if (!response.ok) throw new Error("Export failed");
      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `usuarios_${new Date().toISOString().split("T")[0]}.xlsx`;
      document.body.appendChild(a);
      a.click();
      window.URL.revokeObjectURL(url);
      document.body.removeChild(a);
    },
    onSuccess: () => {
      toast({ title: "Usuários exportados com sucesso!" });
    },
    onError: () => {
      toast({ title: "Erro ao exportar usuários", variant: "destructive" });
    },
  });

  const filteredUsers = users?.filter((user) => {
    const matchesSearch =
      user.email?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.firstName?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      user.lastName?.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesRole = roleFilter === "all" || user.role === roleFilter;
    return matchesSearch && matchesRole;
  });

  if (usersLoading) {
    return <div className="flex items-center justify-center min-h-screen">Carregando...</div>;
  }

  return (
    <div className="space-y-4 sm:space-y-6 lg:space-y-8">
      <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
        <div>
          <h1 className="text-2xl sm:text-3xl lg:text-4xl font-serif font-semibold mb-2">Usuários</h1>
          <p className="text-sm sm:text-base text-muted-foreground">Gerencie usuários e permissões</p>
        </div>
        <Button
          onClick={() => exportMutation.mutate()}
          disabled={exportMutation.isPending}
          data-testid="button-export-users"
          className="w-full sm:w-auto"
        >
          <Download className="w-4 h-4 mr-2" />
          Exportar XLSX
        </Button>
      </div>

      {/* Search and Filters */}
      <Card>
        <CardContent className="p-4 sm:p-6">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <Input
                placeholder="Buscar por nome ou email..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 text-sm sm:text-base"
                data-testid="input-search-users"
              />
            </div>
            <Select value={roleFilter} onValueChange={setRoleFilter}>
              <SelectTrigger className="w-full sm:w-48" data-testid="select-role-filter">
                <SelectValue placeholder="Filtrar por role" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">Todos</SelectItem>
                <SelectItem value="admin">Administradores</SelectItem>
                <SelectItem value="student">Alunos</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </CardContent>
      </Card>

      {/* Users Table */}
      <Card>
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="min-w-[120px] sm:min-w-[150px]">Usuário</TableHead>
                  <TableHead className="min-w-[120px] sm:min-w-[150px]">Email</TableHead>
                  <TableHead className="min-w-[70px] sm:min-w-[80px]">Role</TableHead>
                  <TableHead className="min-w-[60px] sm:min-w-[80px]">Pontos</TableHead>
                  <TableHead className="min-w-[80px] sm:min-w-[100px]">Cadastro</TableHead>
                  <TableHead className="text-right min-w-[100px] sm:min-w-[120px]">Ações</TableHead>
                </TableRow>
              </TableHeader>
            <TableBody>
              {filteredUsers?.map((user) => (
                <TableRow key={user.id} data-testid={`row-user-${user.id}`}>
                  <TableCell>
                    <div className="flex items-center gap-2 sm:gap-3">
                      <Avatar className="w-8 h-8 sm:w-10 sm:h-10 flex-shrink-0">
                        <AvatarImage src={user.profileImageUrl || ""} />
                        <AvatarFallback>
                          {user.firstName?.[0]}{user.lastName?.[0]}
                        </AvatarFallback>
                      </Avatar>
                      <div className="min-w-0">
                        <p className="font-medium text-sm sm:text-base truncate">
                          {user.firstName} {user.lastName}
                        </p>
                        <p className="text-xs text-muted-foreground hidden sm:block">ID: {user.id.slice(0, 8)}...</p>
                      </div>
                    </div>
                  </TableCell>
                  <TableCell className="text-muted-foreground text-sm sm:text-base">
                    <span className="truncate block max-w-[200px] sm:max-w-none">{user.email}</span>
                  </TableCell>
                  <TableCell>
                    <Badge variant={user.role === "admin" ? "default" : "secondary"}>
                      {user.role === "admin" ? (
                        <><Shield className="w-3 h-3 mr-1" /> Admin</>
                      ) : (
                        <><GraduationCap className="w-3 h-3 mr-1" /> Aluno</>
                      )}
                    </Badge>
                  </TableCell>
                  <TableCell className="font-mono text-sm sm:text-base">{user.points}</TableCell>
                  <TableCell className="text-muted-foreground text-sm sm:text-base">
                    {user.createdAt ? new Date(user.createdAt).toLocaleDateString("pt-BR") : "-"}
                  </TableCell>
                  <TableCell className="text-right">
                    <Dialog 
                      open={isGrantAccessDialogOpen && selectedUser?.id === user.id}
                      onOpenChange={(open) => {
                        setIsGrantAccessDialogOpen(open);
                        if (open) {
                          setSelectedUser(user);
                          setSelectedCourses([]);
                        } else {
                          setSelectedUser(null);
                          setSelectedCourses([]);
                        }
                      }}
                    >
                      <DialogTrigger asChild>
                        <Button 
                          size="sm" 
                          variant="outline"
                          data-testid={`button-grant-access-${user.id}`}
                          className="w-full sm:w-auto"
                        >
                          <BookOpen className="w-4 h-4 mr-2" />
                          <span className="hidden sm:inline">Atribuir Cursos</span>
                          <span className="sm:hidden">Cursos</span>
                        </Button>
                      </DialogTrigger>
                      <DialogContent>
                        <DialogHeader>
                          <DialogTitle>Atribuir Cursos</DialogTitle>
                          <DialogDescription>
                            Selecione os cursos para atribuir a {user.firstName} {user.lastName}
                          </DialogDescription>
                        </DialogHeader>
                        <div className="space-y-4 max-h-96 overflow-y-auto">
                          {courses?.map((course) => (
                            <div key={course.id} className="flex items-center space-x-2">
                              <Checkbox
                                id={`course-${course.id}`}
                                checked={selectedCourses.includes(course.id)}
                                onCheckedChange={(checked) => {
                                  if (checked) {
                                    setSelectedCourses([...selectedCourses, course.id]);
                                  } else {
                                    setSelectedCourses(selectedCourses.filter(id => id !== course.id));
                                  }
                                }}
                                data-testid={`checkbox-course-${course.id}`}
                              />
                              <label
                                htmlFor={`course-${course.id}`}
                                className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70 cursor-pointer"
                              >
                                {course.title}
                              </label>
                            </div>
                          ))}
                          {!courses?.length && (
                            <p className="text-sm text-muted-foreground text-center py-4">
                              Nenhum curso disponível
                            </p>
                          )}
                        </div>
                        <DialogFooter>
                          <Button
                            onClick={() => grantAccessMutation.mutate()}
                            disabled={grantAccessMutation.isPending}
                            data-testid="button-confirm-grant-access"
                          >
                            {selectedCourses.length === 0 ? "Remover Todos" : `Atribuir (${selectedCourses.length})`}
                          </Button>
                        </DialogFooter>
                      </DialogContent>
                    </Dialog>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
            </Table>
          </div>
          {!filteredUsers?.length && (
            <div className="text-center py-12 text-muted-foreground">
              <UserPlus className="w-12 h-12 mx-auto mb-4 opacity-50" />
              <p>Nenhum usuário encontrado</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
