// Admin Support - Support tickets and FAQ management
import { useState, useEffect } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Accordion,
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from "@/components/ui/accordion";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { HelpCircle, MessageSquare } from "lucide-react";
import type { SupportTicket, FaqItem } from "@shared/schema";

export default function AdminSupport() {
  const { isAuthenticated, isLoading } = useAuth();
  const { toast } = useToast();

  // Redirect if not authenticated
  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: tickets } = useQuery<SupportTicket[]>({
    queryKey: ["/api/admin/support/tickets"],
    enabled: isAuthenticated,
  });

  const { data: faqs } = useQuery<FaqItem[]>({
    queryKey: ["/api/admin/faq"],
    enabled: isAuthenticated,
  });

  const getStatusVariant = (status: string): "default" | "secondary" | "outline" => {
    switch (status) {
      case "open":
        return "default";
      case "in_progress":
        return "secondary";
      case "resolved":
        return "outline";
      default:
        return "secondary";
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case "urgent":
        return "text-destructive";
      case "high":
        return "text-chart-4";
      case "medium":
        return "text-chart-1";
      default:
        return "text-muted-foreground";
    }
  };

  return (
    <div className="space-y-4 sm:space-y-6 lg:space-y-8">
      <div>
        <h1 className="text-2xl sm:text-3xl lg:text-4xl font-serif font-semibold mb-2">Atendimento</h1>
        <p className="text-sm sm:text-base text-muted-foreground">Gerencie tickets de suporte e FAQ</p>
      </div>

      <Tabs defaultValue="tickets" className="space-y-6">
        <TabsList className="w-full sm:w-auto">
          <TabsTrigger value="tickets" data-testid="tab-tickets" className="flex-1 sm:flex-initial">
            <MessageSquare className="w-4 h-4 mr-2" />
            <span className="hidden sm:inline">Suporte</span>
            <span className="sm:hidden">Tickets</span>
          </TabsTrigger>
          <TabsTrigger value="faq" data-testid="tab-faq" className="flex-1 sm:flex-initial">
            <HelpCircle className="w-4 h-4 mr-2" />
            FAQ
          </TabsTrigger>
        </TabsList>

        <TabsContent value="tickets" className="space-y-6">
          <Card>
                <CardContent className="p-0">
              <div className="overflow-x-auto">
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead className="min-w-[120px] sm:min-w-[150px]">Assunto</TableHead>
                      <TableHead className="min-w-[120px] sm:min-w-[150px]">Email</TableHead>
                      <TableHead className="min-w-[80px] sm:min-w-[100px]">Prioridade</TableHead>
                      <TableHead className="min-w-[80px] sm:min-w-[100px]">Status</TableHead>
                      <TableHead className="min-w-[80px] sm:min-w-[100px]">Data</TableHead>
                    </TableRow>
                  </TableHeader>
                <TableBody>
                  {tickets?.map((ticket) => (
                    <TableRow key={ticket.id} data-testid={`row-ticket-${ticket.id}`}>
                      <TableCell className="font-medium text-sm sm:text-base">
                        <span className="truncate block max-w-[150px] sm:max-w-none">{ticket.subject}</span>
                      </TableCell>
                      <TableCell className="text-muted-foreground text-sm sm:text-base">
                        <span className="truncate block max-w-[150px] sm:max-w-none">{ticket.userEmail}</span>
                      </TableCell>
                      <TableCell>
                        <span className={`text-xs sm:text-sm font-medium ${getPriorityColor(ticket.priority)}`}>
                          {ticket.priority.toUpperCase()}
                        </span>
                      </TableCell>
                      <TableCell>
                        <Badge variant={getStatusVariant(ticket.status)} className="text-xs sm:text-sm">
                          <span className="hidden sm:inline">
                            {ticket.status === "open"
                              ? "Aberto"
                              : ticket.status === "in_progress"
                              ? "Em Andamento"
                              : ticket.status === "resolved"
                              ? "Resolvido"
                              : "Fechado"}
                          </span>
                          <span className="sm:hidden">
                            {ticket.status === "open"
                              ? "Aberto"
                              : ticket.status === "in_progress"
                              ? "Andamento"
                              : ticket.status === "resolved"
                              ? "Resolvido"
                              : "Fechado"}
                          </span>
                        </Badge>
                      </TableCell>
                      <TableCell className="text-muted-foreground text-xs sm:text-sm">
                        {ticket.createdAt ? new Date(ticket.createdAt).toLocaleDateString("pt-BR") : "-"}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
                </Table>
              </div>
              {!tickets?.length && (
                <div className="text-center py-12 text-muted-foreground">
                  <MessageSquare className="w-12 h-12 mx-auto mb-4 opacity-50" />
                  <p>Nenhum ticket de suporte</p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="faq" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="text-lg sm:text-xl">Perguntas Frequentes</CardTitle>
            </CardHeader>
            <CardContent className="p-4 sm:p-6">
              <Accordion type="single" collapsible className="w-full">
                {faqs?.map((faq, index) => (
                  <AccordionItem key={faq.id} value={`item-${index}`} data-testid={`faq-${index}`}>
                    <AccordionTrigger className="text-left">
                      {faq.question}
                    </AccordionTrigger>
                    <AccordionContent className="text-muted-foreground">
                      {faq.answer}
                    </AccordionContent>
                  </AccordionItem>
                ))}
              </Accordion>
              {!faqs?.length && (
                <div className="text-center py-12 text-muted-foreground">
                  <HelpCircle className="w-12 h-12 mx-auto mb-4 opacity-50" />
                  <p>Nenhuma pergunta no FAQ</p>
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
