// Admin Sales - Sales from webhook
import { useEffect } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { DollarSign, CheckCircle, Clock, XCircle } from "lucide-react";
import type { Sale } from "@shared/schema";

export default function AdminSales() {
  const { isAuthenticated, isLoading } = useAuth();
  const { toast } = useToast();

  // Redirect if not authenticated
  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: sales, isLoading: salesLoading } = useQuery<Sale[]>({
    queryKey: ["/api/admin/sales"],
    enabled: isAuthenticated,
  });

  if (salesLoading) {
    return <div className="flex items-center justify-center min-h-screen">Carregando...</div>;
  }

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "processed":
        return <CheckCircle className="w-4 h-4" />;
      case "pending":
        return <Clock className="w-4 h-4" />;
      case "error":
        return <XCircle className="w-4 h-4" />;
      default:
        return null;
    }
  };

  const getStatusVariant = (status: string): "default" | "secondary" | "destructive" => {
    switch (status) {
      case "processed":
        return "default";
      case "pending":
        return "secondary";
      case "error":
        return "destructive";
      default:
        return "secondary";
    }
  };

  return (
    <div className="space-y-4 sm:space-y-6 lg:space-y-8">
      <div>
        <h1 className="text-2xl sm:text-3xl lg:text-4xl font-serif font-semibold mb-2">Vendas</h1>
        <p className="text-sm sm:text-base text-muted-foreground">Vendas recebidas via webhook da plataforma de pagamento</p>
      </div>

      {/* Sales Table */}
      <Card>
        <CardContent className="p-0">
          <div className="overflow-x-auto">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="min-w-[120px] sm:min-w-[150px]">Data/Hora</TableHead>
                  <TableHead className="min-w-[120px] sm:min-w-[150px]">Email Comprador</TableHead>
                  <TableHead className="min-w-[100px] sm:min-w-[150px]">Curso</TableHead>
                  <TableHead className="min-w-[80px] sm:min-w-[100px]">Valor</TableHead>
                  <TableHead className="min-w-[80px] sm:min-w-[100px]">Status</TableHead>
                  <TableHead className="min-w-[100px] sm:min-w-[150px]">Webhook ID</TableHead>
                </TableRow>
              </TableHeader>
            <TableBody>
              {sales?.map((sale) => (
                <TableRow key={sale.id} data-testid={`row-sale-${sale.id}`}>
                  <TableCell className="font-mono text-xs sm:text-sm">
                    {new Date(sale.purchasedAt).toLocaleString("pt-BR")}
                  </TableCell>
                  <TableCell className="text-muted-foreground text-sm sm:text-base">
                    <span className="truncate block max-w-[150px] sm:max-w-none">{sale.buyerEmail}</span>
                  </TableCell>
                  <TableCell>
                    <span className="text-xs sm:text-sm">{sale.courseId ? `Curso ID: ${sale.courseId.slice(0, 8)}...` : "-"}</span>
                  </TableCell>
                  <TableCell className="font-mono font-medium text-sm sm:text-base">
                    {sale.amount ? `${sale.currency} ${Number(sale.amount).toFixed(2)}` : "-"}
                  </TableCell>
                  <TableCell>
                    <Badge variant={getStatusVariant(sale.status)} className="gap-1 text-xs sm:text-sm">
                      {getStatusIcon(sale.status)}
                      <span className="hidden sm:inline">
                        {sale.status === "processed" ? "Processado" : sale.status === "pending" ? "Pendente" : "Erro"}
                      </span>
                      <span className="sm:hidden">
                        {sale.status === "processed" ? "OK" : sale.status === "pending" ? "..." : "!"}
                      </span>
                    </Badge>
                  </TableCell>
                  <TableCell className="font-mono text-xs text-muted-foreground">
                    <span className="truncate block max-w-[100px] sm:max-w-none">{sale.webhookId || "-"}</span>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
            </Table>
          </div>
          {!sales?.length && (
            <div className="text-center py-12 text-muted-foreground">
              <DollarSign className="w-12 h-12 mx-auto mb-4 opacity-50" />
              <p>Nenhuma venda registrada</p>
              <p className="text-sm mt-2">As vendas aparecerão aqui quando recebidas via webhook</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
