import { useState, useRef } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { ArrowLeft, UploadCloud, Eye, Save, FileText } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { RichTextEditor } from "@/components/rich-text-editor";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";

interface RouteParams {
  courseId?: string;
  moduleId?: string;
}

export default function AdminLessonNew() {
  const { courseId, moduleId } = useParams<RouteParams>();
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  const [title, setTitle] = useState("");
  const [mediaUrl, setMediaUrl] = useState("");
  const [content, setContent] = useState("");
  const [summary, setSummary] = useState("");
  const [status, setStatus] = useState<"published" | "unlisted">("published");
  const [format, setFormat] = useState("padrao");
  const [thumbnailPreview, setThumbnailPreview] = useState<string>("");
  const [thumbnailFile, setThumbnailFile] = useState<File | null>(null);
  const thumbnailInputRef = useRef<HTMLInputElement | null>(null);
  const materialInputRef = useRef<HTMLInputElement | null>(null);
  const [materialFile, setMaterialFile] = useState<File | null>(null);

  const { data: modulesData, isLoading: modulesLoading } = useQuery<any[]>({
    queryKey: ["course-modules", courseId],
    enabled: Boolean(courseId),
    queryFn: async () => {
      const res = await fetch(`/api/admin/courses/${courseId}/modules`, { credentials: "include" });
      if (!res.ok) throw new Error("Failed to fetch modules");
      return res.json();
    },
  });

  const { data: lessonsData, isLoading: lessonsLoading } = useQuery<any[]>({
    queryKey: ["module-lessons", moduleId],
    enabled: Boolean(moduleId),
    queryFn: async () => {
      const res = await fetch(`/api/admin/modules/${moduleId}/lessons`, { credentials: "include" });
      if (!res.ok) throw new Error("Failed to fetch lessons");
      return res.json();
    },
  });

  const moduleRecord = modulesData?.find((item) => item.id === moduleId);
  const nextOrderIndex = lessonsData?.length ?? 0;

  const createLessonMutation = useMutation({
    mutationFn: async (formData: FormData) => {
      const res = await fetch(`/api/admin/modules/${moduleId}/lessons`, {
        method: "POST",
        credentials: "include",
        body: formData,
      });
      if (!res.ok) throw new Error("Failed to create lesson");
      return res.json();
    },
    onSuccess: async (lesson: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses", courseId, "modules"] });
      queryClient.invalidateQueries({ queryKey: ["course-modules", courseId] });
      queryClient.invalidateQueries({ queryKey: ["module-lessons", moduleId] });
      if (lesson?.id && materialFile) {
        const materialForm = new FormData();
        materialForm.append("title", `${title.trim()} - Material`.slice(0, 255));
        materialForm.append("file", materialFile);

        const materialRes = await fetch(`/api/admin/lessons/${lesson.id}/materials`, {
          method: "POST",
          credentials: "include",
          body: materialForm,
        });
        if (!materialRes.ok) {
          toast({
            title: "Aula salva, mas houve erro ao subir o PDF",
            description: (await materialRes.text()) || "Confira o servidor",
            variant: "destructive",
          });
        }
      }

      setTitle("");
      setMediaUrl("");
      setContent("");
      setSummary("");
      setStatus("published");
      setFormat("padrao");
      setThumbnailFile(null);
      setThumbnailPreview("");
      setMaterialFile(null);

      toast({ title: "Aula criada com sucesso!" });
      if (courseId) {
        setLocation(`/admin/courses/${courseId}/content`);
      }
    },
    onError: () => {
      toast({ title: "Erro ao criar aula", variant: "destructive" });
    },
  });

  const handleThumbnailSelection = (file: File | null) => {
    setThumbnailFile(file);
    if (!file) {
      setThumbnailPreview("");
      return;
    }

    const reader = new FileReader();
    reader.onload = () => {
      const result = reader.result as string;
      setThumbnailPreview(result || "");
    };
    reader.readAsDataURL(file);
  };

  const handleSubmit = (event: React.FormEvent<HTMLFormElement>) => {
    event.preventDefault();
    if (!courseId || !moduleId) return;

    if (!title.trim()) {
      toast({ title: "Informe o título da aula", variant: "destructive" });
      return;
    }

    if (!mediaUrl.trim() && !thumbnailFile && !materialFile) {
      toast({
        title: "Informe pelo menos uma miniatura ou PDF",
        variant: "destructive",
      });
      return;
    }

    if (mediaUrl.trim() && !/^https?:\/\//i.test(mediaUrl.trim())) {
      toast({ title: "Use uma URL de mídia válida iniciando com http(s)", variant: "destructive" });
      return;
    }

    if (!thumbnailFile) {
      toast({ title: "Envie uma miniatura para a aula", variant: "destructive" });
      return;
    }

    const formData = new FormData();
    formData.append("title", title.trim());
    formData.append("status", status);
    formData.append("orderIndex", String(nextOrderIndex));
    if (thumbnailFile) {
      formData.append("thumbnail", thumbnailFile);
    }

    const combinedDescription = summary.trim() || content.trim();
    if (combinedDescription) {
      formData.append("description", combinedDescription);
    }

    formData.append("videoUrl", mediaUrl.trim());

    if (content.trim()) {
      formData.append("content", content.trim());
    }

    if (summary.trim()) {
      formData.append("summary", summary.trim());
    }

    formData.append("format", format);

    createLessonMutation.mutate(formData);
  };

  const isLoading = modulesLoading || lessonsLoading;

  if (isLoading) {
    return <div className="flex min-h-screen items-center justify-center">Carregando...</div>;
  }

  if (!moduleRecord || !courseId || !moduleId) {
    return (
      <div className="flex min-h-screen flex-col items-center justify-center gap-2 text-center">
        <p className="text-lg font-semibold">Módulo não encontrado.</p>
        <Button variant="link" onClick={() => setLocation(`/admin/courses/${courseId}/content`)}>
          Voltar para o curso
        </Button>
      </div>
    );
  }

  return (
    <div className="space-y-8">
      <div className="flex flex-wrap items-center justify-between gap-4">
        <div className="flex items-center gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setLocation(`/admin/courses/${courseId}/content`)}
          >
            <ArrowLeft className="h-5 w-5" />
          </Button>
          <div>
            <h1 className="text-3xl font-serif font-semibold">Novo conteúdo</h1>
            <p className="text-sm text-muted-foreground mt-1">
              {moduleRecord.title} | {moduleRecord.description ? "Personalizado" : "Padrão"}
            </p>
          </div>
        </div>
        <div className="flex items-center gap-2">
          <Button
            variant="secondary"
            className="gap-2"
            onClick={() => toast({ title: "Função de rascunho em breve" })}
          >
            <Eye className="h-4 w-4" /> Pré-visualizar
          </Button>
          <Button
            variant="outline"
            onClick={() => toast({ title: "Salvar texto em rascunho em breve" })}
          >
            Salvar Texto
          </Button>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="grid gap-6 lg:grid-cols-[minmax(0,2fr)_minmax(0,1.2fr)]">
          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="lesson-title">Título *</Label>
              <Input
                id="lesson-title"
                value={title}
                onChange={(event) => setTitle(event.target.value)}
                placeholder="Novo conteúdo"
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="lesson-media-url">Mídia URL (opcional)</Label>
              <Input
                id="lesson-media-url"
                type="url"
                value={mediaUrl}
                onChange={(event) => setMediaUrl(event.target.value)}
                placeholder="Insira o link completo com https://"
              />
              <p className="text-xs text-muted-foreground">
                Compatível com Vimeo, Vadoo, Wistia, YouTube, Loom, Gumlet e outras plataformas.
              </p>
            </div>
            <div className="space-y-2">
              <Label htmlFor="lesson-content">Conteúdo</Label>
              <RichTextEditor
                value={content}
                onChange={setContent}
                placeholder="Cole aqui o roteiro da aula, links úteis ou observações adicionais."
              />
            </div>
          </div>

          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="lesson-status">Status</Label>
              <select
                id="lesson-status"
                value={status}
                onChange={(event) => setStatus(event.target.value as "published" | "unlisted")}
                className="h-10 w-full rounded-md border border-input bg-background px-3 text-sm outline-none focus:ring-2 focus:ring-primary"
              >
                <option value="published">Publicado</option>
                <option value="unlisted">Não listado</option>
              </select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="lesson-format">Formato</Label>
              <select
                id="lesson-format"
                value={format}
                onChange={(event) => setFormat(event.target.value)}
                className="h-10 w-full rounded-md border border-input bg-background px-3 text-sm outline-none focus:ring-2 focus:ring-primary"
              >
                <option value="padrao">Padrão</option>
                <option value="video">Vídeo</option>
                <option value="audio">Áudio</option>
                <option value="texto">Texto</option>
              </select>
            </div>
            <div className="space-y-2">
              <Label htmlFor="lesson-summary">Resumo</Label>
              <Textarea
                id="lesson-summary"
                rows={4}
                placeholder="Resumo rápido sobre o que o aluno verá nesta aula."
                value={summary}
                onChange={(event) => setSummary(event.target.value)}
              />
            </div>
            <div className="space-y-3">
              <Label>Miniatura *</Label>
              <input
                ref={thumbnailInputRef}
                type="file"
                accept="image/*"
                className="hidden"
                onChange={(event) => handleThumbnailSelection(event.target.files?.[0] ?? null)}
              />
              <div className="flex items-center gap-3">
                <Button
                  type="button"
                  variant="secondary"
                  size="sm"
                  className="gap-2"
                  onClick={() => thumbnailInputRef.current?.click()}
                >
                  <UploadCloud className="h-4 w-4" /> Enviar
                </Button>
                {thumbnailFile && (
                  <span className="text-xs text-muted-foreground">{thumbnailFile.name}</span>
                )}
              </div>
              {thumbnailPreview && (
                <div className="overflow-hidden rounded-xl border bg-muted">
                  <img src={thumbnailPreview} alt="Pré-visualização da miniatura" className="h-40 w-full object-cover" />
                </div>
              )}
            </div>
            <div className="space-y-3">
              <Label>Material em PDF (opcional)</Label>
              <input
                ref={materialInputRef}
                type="file"
                accept="application/pdf"
                className="hidden"
                onChange={(event) => setMaterialFile(event.target.files?.[0] ?? null)}
              />
              <div className="flex items-center gap-3">
                <Button
                  type="button"
                  variant="secondary"
                  size="sm"
                  className="gap-2"
                  onClick={() => materialInputRef.current?.click()}
                >
                  <UploadCloud className="h-4 w-4" /> Anexar PDF
                </Button>
                {materialFile ? (
                  <span className="text-xs text-muted-foreground">
                    {materialFile.name} ({(materialFile.size / 1024 / 1024).toFixed(2)} MB)
                  </span>
                ) : (
                  <span className="text-xs text-muted-foreground flex items-center gap-1">
                    <FileText className="h-3 w-3" /> Nenhum PDF selecionado
                  </span>
                )}
              </div>
            </div>
          </div>
        </div>

        <div className="flex flex-wrap items-center justify-between gap-3">
          <Button
            type="button"
            variant="outline"
            size="sm"
            className="gap-2"
            onClick={() => toast({ title: "Função de pré-visualização em breve" })}
          >
            <Eye className="h-4 w-4" /> Pré-visualizar
          </Button>
          <div className="flex items-center gap-2">
            <Button type="button" variant="ghost" onClick={() => setLocation(`/admin/courses/${courseId}/content`)}>
              Cancelar
            </Button>
            <Button type="submit" disabled={createLessonMutation.isPending} className="gap-2">
              {createLessonMutation.isPending ? "Salvando..." : "Salvar aula"}
              <Save className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </form>
    </div>
  );
}
