// Admin Dashboard - Power BI style with charts and metrics
import { useEffect, useMemo } from "react";
import { useAuth } from "@/hooks/useAuth";
import { useToast } from "@/hooks/use-toast";
import { useQuery } from "@tanstack/react-query";
import { MetricCard } from "@/components/metric-card";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Badge } from "@/components/ui/badge";
import {
  Users,
  Award,
  AlertCircle,
  Activity,
  TrendingUp,
  Trophy,
} from "lucide-react";
import {
  LineChart,
  Line,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  Legend,
} from "recharts";

interface DashboardData {
  metrics: {
    totalStudents: number;
    certificatesIssued: number;
    invalidEmails: number;
    accessLast30Days: number;
  };
  studentsByMonth: Array<{ yearMonth: string; month: string; count: number }>;
  accessSegmentation: Array<{ period: string; count: number }>;
  pointsRanking: Array<{
    userId: string;
    name: string;
    email: string;
    points: number;
    profileImageUrl?: string;
  }>;
  engagementRanking: Array<{
    userId: string;
    name: string;
    courseName: string;
    progress: number;
    lastActivity: string;
  }>;
}

export default function AdminDashboard() {
  const { isAuthenticated, isLoading, isAdmin, user } = useAuth();
  const { toast } = useToast();

  // Redirect if not authenticated
  useEffect(() => {
    if (!isLoading && !isAuthenticated) {
      toast({
        title: "Unauthorized",
        description: "You are logged out. Logging in again...",
        variant: "destructive",
      });
      setTimeout(() => {
        window.location.href = "/api/login";
      }, 500);
    }
  }, [isAuthenticated, isLoading, toast]);

  const { data: dashboardData, isLoading: dataLoading } = useQuery<DashboardData>({
    queryKey: ["/api/admin/dashboard"],
    enabled: isAuthenticated && isAdmin,
  });

  // Fake data for demo/client presentation when API has no data
  const DEFAULT_AVATAR = "https://cdn.areademembros.com/cache/default-50x50-cropped.png";

  const mockData: DashboardData = useMemo(() => ({
    metrics: {
      totalStudents: 12450,
      certificatesIssued: 3820,
      invalidEmails: 17,
      accessLast30Days: 8421,
    },
    studentsByMonth: Array.from({ length: 12 }).map((_, idx) => {
      const base = new Date();
      base.setMonth(base.getMonth() - (11 - idx));
      const year = base.getFullYear();
      const monthNum = base.getMonth();
      const monthLabel = base.toLocaleString("pt-BR", { month: "short" }).replace(".", "");
      // simple wave for demo
      const count = 200 + Math.round(80 * Math.sin(idx / 2) + Math.random() * 40);
      return {
        yearMonth: `${year}-${String(monthNum + 1).padStart(2, "0")}`,
        month: monthLabel,
        count,
      };
    }),
    accessSegmentation: [
      { period: "Manhã", count: 3120 },
      { period: "Tarde", count: 3560 },
      { period: "Noite", count: 1741 },
    ],
    pointsRanking: Array.from({ length: 12 }).map((_, i) => ({
      userId: `u${i + 1}`,
      name: [
        "Ana Souza",
        "Bruno Lima",
        "Carla Mendes",
        "Diego Alves",
        "Eduarda Nunes",
        "Felipe Rocha",
        "Gabriela Reis",
        "Henrique Silva",
        "Isabela Freitas",
        "João Pedro",
        "Karla Dias",
        "Lucas Teixeira",
      ][i],
      email: `user${i + 1}@demo.com`,
      points: 1200 - i * 45,
      profileImageUrl: DEFAULT_AVATAR,
    })),
    engagementRanking: Array.from({ length: 10 }).map((_, i) => ({
      userId: `e${i + 1}`,
      name: [
        "Mariana Costa",
        "Natan Farias",
        "Olívia Prado",
        "Paulo Sérgio",
        "Queila Ramos",
        "Rafael Gomes",
        "Sabrina Rocha",
        "Thiago Araujo",
        "Ubiratan Lopes",
        "Vitória Moraes",
      ][i],
      courseName: [
        "Excel Avançado",
        "Power BI Essencial",
        "JavaScript do Zero",
        "Node.js para APIs",
        "UX/UI Fundamentos",
        "Python para Dados",
        "React Completo",
        "Gestão de Projetos",
        "Marketing Digital",
        "SQL Prático",
      ][i],
      progress: 10 + i * 8,
      lastActivity: new Date(Date.now() - i * 86400000).toISOString(),
    })),
  }), []);

  // Merge API with mocks: if API provides empty arrays or tiny lists, use mocks
  const data: DashboardData = useMemo(() => {
    if (!dashboardData) return mockData;
    return {
      metrics: {
        totalStudents: dashboardData.metrics?.totalStudents ?? mockData.metrics.totalStudents,
        certificatesIssued: dashboardData.metrics?.certificatesIssued ?? mockData.metrics.certificatesIssued,
        invalidEmails: dashboardData.metrics?.invalidEmails ?? mockData.metrics.invalidEmails,
        accessLast30Days: dashboardData.metrics?.accessLast30Days ?? mockData.metrics.accessLast30Days,
      },
      studentsByMonth: (dashboardData.studentsByMonth?.length ?? 0) >= 2
        ? dashboardData.studentsByMonth
        : mockData.studentsByMonth,
      accessSegmentation: (dashboardData.accessSegmentation?.length ?? 0) >= 3
        ? dashboardData.accessSegmentation
        : mockData.accessSegmentation,
      pointsRanking: (dashboardData.pointsRanking?.length ?? 0) >= 10
        ? dashboardData.pointsRanking.map((u) => ({ ...u, profileImageUrl: u.profileImageUrl || DEFAULT_AVATAR }))
        : mockData.pointsRanking,
      engagementRanking: (dashboardData.engagementRanking?.length ?? 0) >= 8
        ? dashboardData.engagementRanking
        : mockData.engagementRanking,
    };
  }, [dashboardData, mockData]);

  // Sort studentsByMonth chronologically by yearMonth
  const sortedStudentsByMonth = data?.studentsByMonth
    ? [...data.studentsByMonth].sort((a, b) => a.yearMonth.localeCompare(b.yearMonth))
    : [];

  // Access segmentation helpers
  const accessTotals = useMemo(() => {
    const list = data?.accessSegmentation || [];
    const total = list.reduce((acc, i) => acc + i.count, 0);
    return { list, total };
  }, [data]);

  if (isLoading || dataLoading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-pulse space-y-4 w-full max-w-7xl mx-auto px-4">
          <div className="h-32 bg-muted rounded-md"></div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            {[1, 2, 3, 4].map((i) => (
              <div key={i} className="h-32 bg-muted rounded-md"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-4 sm:space-y-6 lg:space-y-8">
      {/* Greeting */}
      <div className="px-2 sm:px-0">
        <h1 className="text-2xl sm:text-3xl font-semibold mb-1">Olá{user?.name ? `, ${user.name}` : ''}!</h1>
        <p className="text-xs sm:text-sm text-muted-foreground">Visão geral dos últimos 30 dias</p>
      </div>

      {/* Main content: Students chart + KPI tiles */}
      <div className="grid grid-cols-1 lg:grid-cols-12 gap-4 sm:gap-6 items-start">
        <div className="lg:col-span-8">
          {/* Students analytics card (big, like screenshot left) */}
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-2">
                <span className="text-base sm:text-lg">Alunos</span>
                <div className="text-xs text-muted-foreground flex items-center gap-2">
                  <span className="inline-flex items-center gap-1">
                    <span className="w-2 h-2 rounded-full bg-[hsl(var(--chart-1))]"></span>
                    Novos
                  </span>
                </div>
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="h-[250px] sm:h-[300px]">
                <ResponsiveContainer width="100%" height="100%">
                  <LineChart data={sortedStudentsByMonth}>
                    <CartesianGrid strokeDasharray="3 3" stroke="hsl(var(--border))" />
                    <XAxis
                      dataKey="yearMonth"
                      stroke="hsl(var(--muted-foreground))"
                      tickFormatter={(value) => {
                        const item = sortedStudentsByMonth.find((d) => d.yearMonth === value);
                        return item?.month || value;
                      }}
                    />
                    <YAxis stroke="hsl(var(--muted-foreground))" />
                    <Tooltip
                      contentStyle={{
                        backgroundColor: "hsl(var(--popover))",
                        border: "1px solid hsl(var(--border))",
                        borderRadius: "6px",
                      }}
                      labelFormatter={(value) => {
                        const item = sortedStudentsByMonth.find((d) => d.yearMonth === value);
                        return item?.month || value;
                      }}
                    />
                    <Line
                      type="monotone"
                      dataKey="count"
                      name="Novos Alunos"
                      stroke="hsl(var(--chart-1))"
                      strokeWidth={3}
                      dot={{ r: 3, fill: "hsl(var(--chart-1))" }}
                      activeDot={{ r: 5 }}
                    />
                  </LineChart>
                </ResponsiveContainer>
              </div>
              <div className="mt-3 text-xs text-muted-foreground">
                <button className="underline hover:text-foreground">Ver relatórios</button>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Right KPI tiles */}
        <div className="lg:col-span-4 grid grid-cols-1 sm:grid-cols-3 lg:grid-cols-1 gap-4">
          <MetricCard title="Certificados emitidos" value={data?.metrics.certificatesIssued || 0} icon={Award} />
          <MetricCard title="interações" value={data?.metrics.accessLast30Days || 0} icon={Activity} />
          <MetricCard title="Emails inválidos" value={data?.metrics.invalidEmails || 0} icon={AlertCircle} />
        </div>
      </div>

      {/* Rankings + Access segmentation (like lower area in screenshot) */}
      <div className="grid grid-cols-1 lg:grid-cols-12 gap-4 sm:gap-6">
        {/* Left: two horizontal rankings stacked */}
        <div className="lg:col-span-8 space-y-4 sm:space-y-6">
          {/* Points Ranking - horizontal avatars with scroll */}
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="flex items-center gap-2 text-base sm:text-lg">
                <Trophy className="w-4 h-4 sm:w-5 sm:h-5" />
                Ranking de pontos
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-xs text-muted-foreground mb-3">Pontos em progressões</div>
              <div className="overflow-x-auto -mx-4 sm:mx-0 px-4 sm:px-0">
                <div className="flex items-center gap-4 sm:gap-6 min-w-max pr-4">
                  {data?.pointsRanking?.slice(0, 10).map((student, index) => (
                    <div key={student.userId} className="flex flex-col items-center w-20 shrink-0">
                      <div className={`w-12 h-12 rounded-full ring-2 ${index === 0 ? 'ring-yellow-400' : index === 1 ? 'ring-gray-300' : index === 2 ? 'ring-red-400' : 'ring-muted' } flex items-center justify-center overflow-hidden bg-muted`}>
                        <Avatar className="w-12 h-12">
                          <AvatarImage src={student.profileImageUrl || DEFAULT_AVATAR} />
                          <AvatarFallback>{student.name.split(' ').map((n) => n[0]).join('')}</AvatarFallback>
                        </Avatar>
                      </div>
                      <div className="mt-2 text-center">
                        <div className="text-xs font-medium truncate">{index + 1}. {student.name.split(' ')[0]}</div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Engagement Ranking - horizontal with progress */}
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-base sm:text-lg">Ranking de engajamento</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-xs text-muted-foreground mb-3">Aulas assistidas por aluno</div>
              <div className="overflow-x-auto -mx-4 sm:mx-0 px-4 sm:px-0">
                <div className="flex items-end gap-4 sm:gap-6 min-w-max pr-4">
                  {data?.engagementRanking?.slice(0, 10).map((item, index) => (
                    <div key={`${item.userId}-${index}`} className="w-28 shrink-0">
                      <div className="flex flex-col items-center">
                        <Avatar className="w-12 h-12 mb-2">
                          <AvatarImage src={DEFAULT_AVATAR} />
                          <AvatarFallback>{item.name.split(' ').map((n) => n[0]).join('')}</AvatarFallback>
                        </Avatar>
                        <div className="w-full bg-muted rounded-full h-2">
                          <div className="bg-primary h-2 rounded-full" style={{ width: `${item.progress}%` }} />
                        </div>
                        <div className="mt-2 text-center text-xs truncate">{item.name.split(' ')[0]}</div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Right: Access panel with distribution */}
        <div className="lg:col-span-4">
          <Card>
            <CardHeader className="pb-2">
              <CardTitle className="text-base sm:text-lg">Acessos</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-2xl sm:text-3xl font-semibold mb-2">{accessTotals.total}</div>
              <div className="text-xs text-muted-foreground mb-4">Nos últimos 30 dias</div>

              {/* Color bar legend (simple) */}
              <div className="flex gap-2 mb-4">
                <div className="h-2 flex-1 rounded-full bg-blue-500" />
                <div className="h-2 flex-1 rounded-full bg-orange-400" />
                <div className="h-2 flex-1 rounded-full bg-red-500" />
              </div>

              <div className="space-y-3">
                {accessTotals.list.map((row) => {
                  const pct = accessTotals.total ? Math.round((row.count / accessTotals.total) * 1000) / 10 : 0;
                  return (
                    <div key={row.period} className="grid grid-cols-8 items-center gap-2 text-sm">
                      <div className="col-span-3 flex items-center gap-2">
                        <span className="w-2 h-2 rounded-full bg-[hsl(var(--chart-2))]"></span>
                        <span>{row.period}</span>
                      </div>
                      <div className="col-span-3">
                        <div className="w-full bg-muted rounded-full h-2">
                          <div className="h-2 rounded-full bg-foreground/70" style={{ width: `${pct}%` }} />
                        </div>
                      </div>
                      <div className="col-span-2 text-right text-muted-foreground">{row.count} · {pct}%</div>
                    </div>
                  );
                })}
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
