import { useState, useEffect } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import {
  ArrowLeft,
  Plus,
  FileText,
  Video,
  Trash2,
  ChevronDown,
  GripVertical,
  EllipsisVertical,
  PenLine,
  ArrowUpRight,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogFooter,
} from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";

type LessonStatus = "published" | "unlisted";

const LESSON_STATUS_LABELS: Record<LessonStatus, string> = {
  published: "Publicado",
  unlisted: "Não listado",
};

export default function AdminCourseContent() {
  const { courseId } = useParams();
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  const [isModuleDialogOpen, setIsModuleDialogOpen] = useState(false);
  const [isLessonEditorOpen, setIsLessonEditorOpen] = useState(false);
  const [editingLessonId, setEditingLessonId] = useState<string | null>(null);
  const [editingLessonModuleId, setEditingLessonModuleId] = useState<string | null>(null);
  const [lessonStatusPendingId, setLessonStatusPendingId] = useState<string | null>(null);
  const [lessonFormValues, setLessonFormValues] = useState({
    title: "",
    description: "",
    thumbnailUrl: "",
    videoUrl: "",
  });
  const [lessonMaterialFile, setLessonMaterialFile] = useState<File | null>(null);
  const [lessonMaterialTitle, setLessonMaterialTitle] = useState("");
  const [lessonMaterialDescription, setLessonMaterialDescription] = useState("");
  const [moduleDeleteTarget, setModuleDeleteTarget] = useState<{ id: string; title: string } | null>(null);
  const [lessonDeleteTarget, setLessonDeleteTarget] = useState<{ id: string; title: string } | null>(null);

  const {
    data: modules,
    isLoading: modulesLoading,
    isError: modulesError,
  } = useQuery<any[]>({
    queryKey: ["/api/admin/courses", courseId, "modules"],
    queryFn: async () => {
      const res = await fetch(`/api/admin/courses/${courseId}/modules`, {
        credentials: "include",
      });
      if (!res.ok) throw new Error("Failed to fetch modules");
      return res.json();
    },
  });

  // Create a single query to fetch all lessons data
  const [modulesWithLessons, setModulesWithLessons] = useState<any[]>([]);

  useEffect(() => {
    if (!modules) return;

    const fetchAllLessons = async () => {
      const modulesData = await Promise.all(
        modules.map(async (module) => {
          try {
            const res = await fetch(`/api/admin/modules/${module.id}/lessons`, {
              credentials: "include",
            });
            const lessons = res.ok ? await res.json() : [];
            return { ...module, lessons };
          } catch {
            return { ...module, lessons: [] };
          }
        })
      );
      setModulesWithLessons(modulesData);
    };

    fetchAllLessons();
  }, [modules]);

  const createModuleMutation = useMutation({
    mutationFn: async (formData: any) => {
      return await apiRequest("POST", `/api/admin/courses/${courseId}/modules`, formData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses", courseId, "modules"] });
      toast({ title: "Módulo criado com sucesso!" });
      setIsModuleDialogOpen(false);
    },
    onError: () => {
      toast({ title: "Erro ao criar módulo", variant: "destructive" });
    },
  });

  const updateLessonStatusMutation = useMutation({
    mutationFn: async ({ lessonId, status }: { lessonId: string; status: LessonStatus }) => {
      await apiRequest("PATCH", `/api/admin/lessons/${lessonId}/status`, { status });
      return status;
    },
    onMutate: ({ lessonId, status }) => {
      setLessonStatusPendingId(lessonId);
      setModulesWithLessons((prev) =>
        prev.map((module) => ({
          ...module,
          lessons: module.lessons?.map((lesson: any) =>
            lesson.id === lessonId ? { ...lesson, status } : lesson,
          ) ?? [],
        })),
      );
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses", courseId, "modules"] });
      toast({ title: "Status da aula atualizado!" });
    },
    onError: () => {
      toast({ title: "Erro ao atualizar status", variant: "destructive" });
      // Rollback by refetching cached data
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses", courseId, "modules"] });
    },
    onSettled: () => {
      setLessonStatusPendingId(null);
    },
  });

  const saveLessonDetailsMutation = useMutation({
    mutationFn: async ({
      lessonId,
      payload,
      material,
    }: {
      lessonId: string;
      payload: {
        title: string;
        description: string | null;
        thumbnailUrl: string | null;
        videoUrl: string | null;
      };
      material?: { file: File; title: string; description: string | null } | null;
    }) => {
      await apiRequest("PATCH", `/api/admin/lessons/${lessonId}/basic`, payload);

      if (material && material.file) {
        const formData = new FormData();
        formData.append("title", material.title || payload.title);
        if (material.description) {
          formData.append("description", material.description);
        }
        formData.append("file", material.file);

        const res = await fetch(`/api/admin/lessons/${lessonId}/materials`, {
          method: "POST",
          credentials: "include",
          body: formData,
        });
        if (!res.ok) throw new Error("Failed to upload material");
      }
    },
    onSuccess: (_, { lessonId, material }) => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses", courseId, "modules"] });
      if (lessonId) {
        queryClient.invalidateQueries({ queryKey: ["/api/admin/lessons", lessonId, "materials"] });
        // refetchLessonMaterials(); // This function is not defined in the original file
      }
      if (!material?.file) {
        toast({ title: "Aula atualizada com sucesso!" });
      } else {
        toast({ title: "Aula e material salvos com sucesso!" });
      }
      setLessonMaterialFile(null);
      setLessonMaterialTitle("");
      setLessonMaterialDescription("");
    },
    onError: () => {
      toast({ title: "Erro ao salvar aula", variant: "destructive" });
    },
  });

  const deleteLessonMaterialMutation = useMutation({
    mutationFn: async ({ lessonId, materialId }: { lessonId: string; materialId: string }) => {
      const res = await fetch(`/api/admin/lessons/${lessonId}/materials/${materialId}`, {
        method: "DELETE",
        credentials: "include",
      });
      if (!res.ok) throw new Error("Failed to delete material");
    },
    onSuccess: (_, { lessonId }) => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/lessons", lessonId, "materials"] });
      // refetchLessonMaterials(); // This function is not defined in the original file
      toast({ title: "Material removido" });
    },
    onError: () => {
      toast({ title: "Erro ao remover material", variant: "destructive" });
    },
  });

  const deleteLessonMutation = useMutation({
    mutationFn: async (lessonId: string) => {
      await apiRequest("DELETE", `/api/admin/lessons/${lessonId}`);
      return lessonId;
    },
    onSuccess: (lessonId) => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses", courseId, "modules"] });
      toast({ title: "Aula removida" });
      setLessonDeleteTarget(null);
      setIsLessonEditorOpen(false);
      setEditingLessonId(null);
      setModulesWithLessons((prev) =>
        prev.map((module) => ({
          ...module,
          lessons: module.lessons?.filter((lesson: any) => lesson.id !== lessonId) ?? [],
        })),
      );
    },
    onError: () => {
      toast({ title: "Erro ao remover aula", variant: "destructive" });
    },
  });

  const deleteModuleMutation = useMutation({
    mutationFn: async (moduleId: string) => {
      await apiRequest("DELETE", `/api/admin/modules/${moduleId}`);
      return moduleId;
    },
    onSuccess: (moduleId) => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/courses", courseId, "modules"] });
      toast({ title: "Módulo removido" });
      setModuleDeleteTarget(null);
      setModulesWithLessons((prev) => prev.filter((module) => module.id !== moduleId));
    },
    onError: () => {
      toast({ title: "Erro ao remover módulo", variant: "destructive" });
    },
  });

  const handleModuleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    const title = ((formData.get("title") as string) || "").trim();
    const descriptionRaw = ((formData.get("description") as string) || "").trim();
    const thumbnailUrlRaw = ((formData.get("thumbnailUrl") as string) || "").trim();

    createModuleMutation.mutate({
      title,
      description: descriptionRaw || null,
      thumbnailUrl: thumbnailUrlRaw,
      orderIndex: modules?.length || 0,
    });
  };

  const handleLessonEditorSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    if (!editingLessonId) return;

    const payload = {
      title: lessonFormValues.title.trim(),
      description: lessonFormValues.description.trim() ? lessonFormValues.description.trim() : null,
      thumbnailUrl: lessonFormValues.thumbnailUrl.trim() || null,
      videoUrl: lessonFormValues.videoUrl.trim() || null,
    };

    const material = lessonMaterialFile
      ? {
          file: lessonMaterialFile,
          title: (lessonMaterialTitle.trim() || lessonFormValues.title || "Material da aula").trim(),
          description: lessonMaterialDescription.trim() ? lessonMaterialDescription.trim() : null,
        }
      : null;

    saveLessonDetailsMutation.mutate({ lessonId: editingLessonId, payload, material });
  };

  const isLoadingPage = modulesLoading;
  const hasError = modulesError;

  const totalModules = modulesWithLessons.length;
  const totalLessons = modulesWithLessons.reduce(
    (acc, module) => acc + (module.lessons?.length ?? 0),
    0,
  );

  const {
    data: lessonMaterialsData,
    isLoading: lessonMaterialsLoading,
    refetch: refetchLessonMaterials,
  } = useQuery<any[]>({
    queryKey: ["lesson-materials", editingLessonId],
    enabled: Boolean(editingLessonId),
    queryFn: async () => {
      if (!editingLessonId) return [];
      const res = await fetch(`/api/admin/lessons/${editingLessonId}/materials`, {
        credentials: "include",
      });
      if (!res.ok) throw new Error("Failed to fetch lesson materials");
      return res.json();
    },
  });

  const lessonMaterials = lessonMaterialsData ?? [];

  const stats = [
    { label: "Módulos", value: totalModules },
    { label: "Aulas cadastradas", value: totalLessons },
  ];

  const editingLessonModule = editingLessonModuleId
    ? modulesWithLessons.find((module) => module.id === editingLessonModuleId) ?? null
    : null;

  const handlePlaceholderAction = (message: string) =>
    toast({ title: message, description: "Função disponível em breve." });

  const resetLessonEditor = () => {
    setEditingLessonId(null);
    setEditingLessonModuleId(null);
    setLessonFormValues({ title: "", description: "", thumbnailUrl: "", videoUrl: "" });
    setLessonMaterialFile(null);
    setLessonMaterialTitle("");
    setLessonMaterialDescription("");
  };

  const openLessonEditor = (module: any, lesson: any) => {
    setEditingLessonId(lesson.id);
    setEditingLessonModuleId(module.id);
    setLessonFormValues({
      title: lesson.title ?? "",
      description: lesson.description ?? "",
      thumbnailUrl: lesson.thumbnailUrl ?? module.thumbnailUrl ?? "",
      videoUrl: lesson.videoUrl ?? "",
    });
    setLessonMaterialFile(null);
    setLessonMaterialTitle("");
    setLessonMaterialDescription("");
    setIsLessonEditorOpen(true);
  };

  if (hasError) {
    return (
      <div className="flex min-h-screen flex-col items-center justify-center gap-2 text-center">
        <p className="text-lg font-semibold">Erro ao carregar dados do curso.</p>
        <p className="text-sm text-muted-foreground">Tente recarregar a página ou volte mais tarde.</p>
      </div>
    );
  }

  if (isLoadingPage) {
    return <div className="flex min-h-screen items-center justify-center">Carregando...</div>;
  }

  return (
    <div className="space-y-10">
      <div className="flex flex-wrap items-center justify-between gap-4">
        <div className="flex items-center gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setLocation("/admin/vitrine")}
            data-testid="button-back"
          >
            <ArrowLeft className="w-5 h-5" />
          </Button>
          <div>
            <h1 className="text-3xl font-serif font-semibold">Gerenciamento do Curso</h1>
            <p className="text-sm text-muted-foreground mt-1">
              Organize módulos, aulas e materiais do produto selecionado.
            </p>
          </div>
        </div>
        <div className="flex items-center gap-2">
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button variant="outline" size="sm" className="gap-2">
                Liberação
                <ChevronDown className="w-4 h-4" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end">
              <DropdownMenuItem
                onSelect={(event) => {
                  event.preventDefault();
                  handlePlaceholderAction("Configurações de liberação");
                }}
              >
                Em breve
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
          <Button
            variant="outline"
            size="sm"
            className="gap-2"
            onClick={() => handlePlaceholderAction("Excluir curso")}
          >
            <Trash2 className="w-4 h-4" />
            Deletar Curso
          </Button>
        </div>
      </div>

      <div className="grid gap-4 sm:grid-cols-2 xl:grid-cols-4">
        {stats.map((stat) => (
          <Card key={stat.label}>
            <CardContent className="py-6">
              <p className="text-sm text-muted-foreground">{stat.label}</p>
              <p className="text-3xl font-semibold mt-2">{stat.value}</p>
            </CardContent>
          </Card>
        ))}
      </div>

      <Tabs defaultValue="content" className="space-y-6">
        <TabsList className="w-full justify-start overflow-auto">
          <TabsTrigger value="content">Conteúdo</TabsTrigger>
          <TabsTrigger value="exams" disabled>
            Provas
          </TabsTrigger>
          <TabsTrigger value="certificate" disabled>
            Certificado
          </TabsTrigger>
          <TabsTrigger value="display" disabled>
            Exibição
          </TabsTrigger>
          <TabsTrigger value="gamification" disabled>
            Gamificação
          </TabsTrigger>
          <TabsTrigger value="settings" disabled>
            Definições
          </TabsTrigger>
        </TabsList>

        <TabsContent value="content" className="space-y-10">
          <div className="space-y-4">
            <div className="flex flex-wrap items-center justify-between gap-3">
              <div>
                <h2 className="text-xl font-semibold">Módulos e aulas</h2>
                <p className="text-sm text-muted-foreground">
                  Controle a ordem e o status das aulas cadastradas.
                </p>
              </div>
              <Dialog open={isModuleDialogOpen} onOpenChange={setIsModuleDialogOpen}>
                <DialogTrigger asChild>
                  <Button size="sm" data-testid="button-add-module" className="gap-2">
                    <Plus className="w-4 h-4" />
                    Novo módulo
                  </Button>
                </DialogTrigger>
                <DialogContent>
                  <DialogHeader>
                    <DialogTitle>Criar módulo</DialogTitle>
                    <DialogDescription>Organize suas aulas por temas.</DialogDescription>
                  </DialogHeader>
                  <form onSubmit={handleModuleSubmit} className="space-y-4">
                    <div>
                      <Label htmlFor="module-title">Título do módulo *</Label>
                      <Input id="module-title" name="title" required data-testid="input-module-title" />
                    </div>
                    <div>
                      <Label htmlFor="module-thumbnail">Thumbnail do módulo *</Label>
                      <Input
                        id="module-thumbnail"
                        name="thumbnailUrl"
                        type="url"
                        required
                        placeholder="https://imagem.com/modulo.png"
                      />
                      <p className="text-xs text-muted-foreground mt-1">
                        Prefira uma imagem quadrada (ex.: 320x320).
                      </p>
                    </div>
                    <div>
                      <Label htmlFor="module-description">Descrição</Label>
                      <Textarea
                        id="module-description"
                        name="description"
                        rows={3}
                        data-testid="input-module-description"
                      />
                    </div>
                    <DialogFooter>
                      <Button type="submit" disabled={createModuleMutation.isPending}>
                        {createModuleMutation.isPending ? "Criando..." : "Criar módulo"}
                      </Button>
                    </DialogFooter>
                  </form>
                </DialogContent>
              </Dialog>
            </div>

            {modulesWithLessons && modulesWithLessons.length > 0 ? (
              <div className="space-y-6">
                {modulesWithLessons.map((module: any) => {
                  const lessonCount = module.lessons?.length ?? 0;

                  return (
                    <div
                      key={module.id}
                      className="rounded-3xl border bg-background p-6 shadow-sm"
                      data-testid={`module-${module.id}`}
                    >
                      <div className="flex flex-wrap items-start justify-between gap-4">
                        <div className="flex items-center gap-4">
                          <GripVertical className="hidden text-muted-foreground sm:block" />
                          <div className="h-16 w-16 overflow-hidden rounded-xl border bg-muted">
                            {module.thumbnailUrl ? (
                              <img
                                src={module.thumbnailUrl}
                                alt={`Thumbnail do módulo ${module.title}`}
                                className="h-full w-full object-cover"
                                onError={(event) => {
                                  event.currentTarget.style.display = "none";
                                }}
                              />
                            ) : (
                              <div className="flex h-full w-full items-center justify-center text-base font-semibold text-muted-foreground">
                                {module.title?.charAt(0)?.toUpperCase() ?? "M"}
                              </div>
                            )}
                          </div>
                          <div className="space-y-1">
                            <div className="flex items-center gap-2">
                              <h3 className="text-lg font-semibold leading-tight">{module.title}</h3>
                            </div>
                            <p className="text-sm text-muted-foreground">
                              {module.description || `${lessonCount} aula${lessonCount === 1 ? "" : "s"} cadastrada${lessonCount === 1 ? "" : "s"}`}
                            </p>
                          </div>
                        </div>
                        <div className="flex items-center gap-2">
                          <Badge className="bg-emerald-50 text-emerald-700 border-emerald-100">Publicado</Badge>
                          <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                              <Button variant="outline" size="sm" className="gap-2">
                                Ações
                                <ChevronDown className="w-4 h-4" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end" className="w-48">
                              <DropdownMenuItem
                                onSelect={(event) => {
                                  event.preventDefault();
                                  handlePlaceholderAction("Renomear módulo");
                                }}
                              >
                                <PenLine className="w-4 h-4 mr-2" /> Renomear módulo
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                onSelect={(event) => {
                                  event.preventDefault();
                                  setLocation(`/admin/courses/${courseId}/modules/${module.id}/lessons/new`);
                                }}
                              >
                                <Plus className="w-4 h-4 mr-2" /> Adicionar aula
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                onSelect={(event) => {
                                  event.preventDefault();
                                  handlePlaceholderAction("Duplicar módulo");
                                }}
                              >
                                Duplicar
                              </DropdownMenuItem>
                              <DropdownMenuItem
                                onSelect={(event) => {
                                  event.preventDefault();
                                  setModuleDeleteTarget({ id: module.id, title: module.title });
                                }}
                                className="text-destructive focus:text-destructive"
                              >
                                <Trash2 className="w-4 h-4 mr-2" /> Remover módulo
                              </DropdownMenuItem>
                            </DropdownMenuContent>
                          </DropdownMenu>
                        </div>
                      </div>

                      {lessonCount > 0 ? (
                        <div className="mt-6 space-y-3">
                          {module.lessons.map((lesson: any) => {
                            const lessonStatus = (lesson.status as LessonStatus) || "published";
                            const isStatusLoading = lessonStatusPendingId === lesson.id;
                            const thumbnailSrc = lesson.thumbnailUrl || module.thumbnailUrl;

                            return (
                              <div
                                key={lesson.id}
                                className="flex flex-col gap-4 rounded-2xl border bg-muted/30 p-4 md:flex-row md:items-center md:justify-between"
                                data-testid={`lesson-${lesson.id}`}
                              >
                                <div className="flex flex-1 items-center gap-4">
                                  <GripVertical className="h-4 w-4 text-muted-foreground" />
                                  <div className="h-12 w-12 overflow-hidden rounded-xl border bg-muted">
                                    {thumbnailSrc ? (
                                      <img
                                        src={thumbnailSrc}
                                        alt={`Thumbnail da aula ${lesson.title}`}
                                        className="h-full w-full object-cover"
                                        onError={(event) => {
                                          event.currentTarget.style.display = "none";
                                        }}
                                      />
                                    ) : (
                                      <div className="flex h-full w-full items-center justify-center text-sm font-semibold text-muted-foreground">
                                        {lesson.title?.charAt(0)?.toUpperCase() ?? "A"}
                                      </div>
                                    )}
                                  </div>
                                  <p className="font-medium text-sm leading-snug flex-1">{lesson.title}</p>
                                </div>

                                <div className="flex flex-col items-stretch gap-2 md:flex-row md:items-center md:gap-3">
                                  <DropdownMenu>
                                    <DropdownMenuTrigger asChild>
                                      <Button
                                        variant="outline"
                                        size="sm"
                                        className="gap-2 rounded-full border px-3"
                                        disabled={isStatusLoading}
                                      >
                                        {LESSON_STATUS_LABELS[lessonStatus]}
                                        <ChevronDown className="w-3 h-3" />
                                      </Button>
                                    </DropdownMenuTrigger>
                                    <DropdownMenuContent align="end">
                                      {(["published", "unlisted"] as LessonStatus[]).map((statusOption) => (
                                        <DropdownMenuItem
                                          key={statusOption}
                                          onSelect={(event) => {
                                            event.preventDefault();
                                            updateLessonStatusMutation.mutate({
                                              lessonId: lesson.id,
                                              status: statusOption,
                                            });
                                          }}
                                          disabled={isStatusLoading || lessonStatus === statusOption}
                                        >
                                          {LESSON_STATUS_LABELS[statusOption]}
                                        </DropdownMenuItem>
                                      ))}
                                    </DropdownMenuContent>
                                  </DropdownMenu>

                                  <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={() => openLessonEditor(module, lesson)}
                                  >
                                    <PenLine className="h-4 w-4" />
                                  </Button>
                                  <Button
                                    variant="ghost"
                                    size="icon"
                                    className="text-destructive"
                                    onClick={() => setLessonDeleteTarget({ id: lesson.id, title: lesson.title })}
                                    disabled={deleteLessonMutation.isPending && lessonDeleteTarget?.id === lesson.id}
                                  >
                                    <Trash2 className="h-4 w-4" />
                                  </Button>
                                </div>
                              </div>
                            );
                          })}
                        </div>
                      ) : (
                        <div className="mt-6 rounded-2xl border border-dashed p-6 text-center text-sm text-muted-foreground">
                          Nenhuma aula cadastrada. Utilize o menu para adicionar uma nova aula.
                        </div>
                      )}

                      <div className="mt-6 flex justify-end">
                        <Button
                          variant="default"
                          size="sm"
                          onClick={() => setLocation(`/admin/courses/${courseId}/modules/${module.id}/lessons/new`)}
                          className="gap-2 rounded-full bg-primary px-6 text-white shadow-md hover:bg-primary/90"
                          data-testid={`button-add-lesson-${module.id}`}
                        >
                          <Plus className="w-4 h-4" /> Nova aula
                        </Button>
                      </div>
                    </div>
                  );
                })}
              </div>
            ) : (
              <div className="rounded-xl border border-dashed p-12 text-center text-muted-foreground">
                <Video className="w-12 h-12 mx-auto mb-4 opacity-50" />
                <p className="font-medium">Nenhum módulo criado</p>
                <p className="text-sm">Clique em "Novo módulo" para começar</p>
              </div>
            )}
          </div>

          <Dialog
            open={isLessonEditorOpen}
            onOpenChange={(open) => {
              setIsLessonEditorOpen(open);
              if (!open) {
                resetLessonEditor();
              }
            }}
          >
            <DialogContent className="max-w-2xl">
              <DialogHeader>
                <DialogTitle>Editar aula</DialogTitle>
                <DialogDescription>
                  {editingLessonModule
                    ? `Atualize as informações da aula vinculada ao módulo "${editingLessonModule.title}".`
                    : "Atualize as informações da aula selecionada."}
                </DialogDescription>
              </DialogHeader>
              {editingLessonId ? (
                <form onSubmit={handleLessonEditorSubmit} className="space-y-5">
                  <div className="grid gap-4 sm:grid-cols-2">
                    <div>
                      <Label htmlFor="edit-lesson-title">Título *</Label>
                      <Input
                        id="edit-lesson-title"
                        value={lessonFormValues.title}
                        onChange={(event) =>
                          setLessonFormValues((prev) => ({ ...prev, title: event.target.value }))
                        }
                        required
                      />
                    </div>
                    <div>
                      <Label htmlFor="edit-lesson-thumbnail">Thumbnail</Label>
                      <Input
                        id="edit-lesson-thumbnail"
                        type="url"
                        placeholder="https://imagem.com/aula.png"
                        value={lessonFormValues.thumbnailUrl}
                        onChange={(event) =>
                          setLessonFormValues((prev) => ({ ...prev, thumbnailUrl: event.target.value }))
                        }
                      />
                    </div>
                  </div>
                  <div>
                    <Label htmlFor="edit-lesson-description">Descrição</Label>
                    <Textarea
                      id="edit-lesson-description"
                      rows={3}
                      value={lessonFormValues.description}
                      onChange={(event) =>
                        setLessonFormValues((prev) => ({ ...prev, description: event.target.value }))
                      }
                    />
                  </div>
                  <div>
                    <Label htmlFor="edit-lesson-video-url">Link do vídeo</Label>
                    <Input
                      id="edit-lesson-video-url"
                      type="url"
                      placeholder="https://player.vimeo.com/..."
                      value={lessonFormValues.videoUrl}
                      onChange={(event) =>
                        setLessonFormValues((prev) => ({ ...prev, videoUrl: event.target.value }))
                      }
                    />
                    <p className="text-xs text-muted-foreground mt-1">
                      Deixe em branco para continuar utilizando o vídeo enviado diretamente.
                    </p>
                  </div>
                  <div className="space-y-3">
                    <div>
                      <h4 className="text-sm font-semibold">Materiais em PDF</h4>
                      <p className="text-xs text-muted-foreground">
                        Anexe arquivos de apoio que ficarão disponíveis dentro desta aula.
                      </p>
                    </div>
                    {lessonMaterialsLoading ? (
                      <p className="text-xs text-muted-foreground">Carregando materiais...</p>
                    ) : lessonMaterials.length > 0 ? (
                      <div className="space-y-2">
                        {lessonMaterials.map((material: any) => (
                          <div
                            key={material.id}
                            className="flex items-center justify-between rounded-lg border bg-background p-3"
                          >
                            <div>
                              <p className="text-sm font-medium">{material.title}</p>
                              <p className="text-xs text-muted-foreground">{material.fileName}</p>
                            </div>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() =>
                                editingLessonId &&
                                deleteLessonMaterialMutation.mutate({
                                  lessonId: editingLessonId,
                                  materialId: material.id,
                                })
                              }
                              disabled={deleteLessonMaterialMutation.isPending}
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        ))}
                      </div>
                    ) : (
                      <p className="text-xs text-muted-foreground">Nenhum material cadastrado.</p>
                    )}
                    <div className="grid gap-3 sm:grid-cols-2">
                      <div>
                        <Label htmlFor="edit-material-title">Título do material</Label>
                        <Input
                          id="edit-material-title"
                          value={lessonMaterialTitle}
                          onChange={(event) => setLessonMaterialTitle(event.target.value)}
                          placeholder="Ex.: Apostila da aula"
                        />
                      </div>
                      <div>
                        <Label htmlFor="edit-material-description">Descrição</Label>
                        <Input
                          id="edit-material-description"
                          value={lessonMaterialDescription}
                          onChange={(event) => setLessonMaterialDescription(event.target.value)}
                          placeholder="Informações adicionais"
                        />
                      </div>
                    </div>
                    <div>
                      <Label htmlFor="edit-material-file">Arquivo PDF</Label>
                      <Input
                        id="edit-material-file"
                        type="file"
                        accept="application/pdf,.pdf"
                        onChange={(event) => setLessonMaterialFile(event.target.files?.[0] ?? null)}
                      />
                      {lessonMaterialFile && (
                        <p className="text-xs text-muted-foreground mt-1">
                          {lessonMaterialFile.name} ({(lessonMaterialFile.size / 1024 / 1024).toFixed(2)} MB)
                        </p>
                      )}
                    </div>
                  </div>
                  <DialogFooter>
                    <Button
                      type="button"
                      variant="outline"
                      onClick={() => {
                        setIsLessonEditorOpen(false);
                        resetLessonEditor();
                      }}
                    >
                      Cancelar
                    </Button>
                    <Button type="submit" disabled={saveLessonDetailsMutation.isPending || lessonFormValues.title.trim() === ""}>
                      {saveLessonDetailsMutation.isPending ? "Salvando..." : "Salvar alterações"}
                    </Button>
                  </DialogFooter>
                </form>
              ) : (
                <div className="py-6 text-sm text-muted-foreground">Selecione uma aula para editar.</div>
              )}
            </DialogContent>
          </Dialog>

        <Dialog
          open={Boolean(moduleDeleteTarget)}
          onOpenChange={(open) => {
            if (!open && !deleteModuleMutation.isPending) {
              setModuleDeleteTarget(null);
            }
          }}
        >
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Remover módulo</DialogTitle>
              <DialogDescription>
                {moduleDeleteTarget
                  ? `Esta ação remove o módulo "${moduleDeleteTarget.title}" e todas as aulas e materiais vinculados. Deseja continuar?`
                  : "Esta ação remove o módulo selecionado e todo o seu conteúdo."}
              </DialogDescription>
            </DialogHeader>
            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => setModuleDeleteTarget(null)}
                disabled={deleteModuleMutation.isPending}
              >
                Cancelar
              </Button>
              <Button
                variant="destructive"
                onClick={() => moduleDeleteTarget && deleteModuleMutation.mutate(moduleDeleteTarget.id)}
                disabled={deleteModuleMutation.isPending}
              >
                {deleteModuleMutation.isPending ? "Removendo..." : "Remover módulo"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        <Dialog
          open={Boolean(lessonDeleteTarget)}
          onOpenChange={(open) => {
            if (!open && !deleteLessonMutation.isPending) {
              setLessonDeleteTarget(null);
            }
          }}
        >
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Remover aula</DialogTitle>
              <DialogDescription>
                {lessonDeleteTarget
                  ? `A aula "${lessonDeleteTarget.title}" e seus materiais serão removidos. Deseja continuar?`
                  : "Esta ação remove a aula selecionada e seus materiais."}
              </DialogDescription>
            </DialogHeader>
            <DialogFooter>
              <Button
                variant="outline"
                onClick={() => setLessonDeleteTarget(null)}
                disabled={deleteLessonMutation.isPending}
              >
                Cancelar
              </Button>
              <Button
                variant="destructive"
                onClick={() => lessonDeleteTarget && deleteLessonMutation.mutate(lessonDeleteTarget.id)}
                disabled={deleteLessonMutation.isPending}
              >
                {deleteLessonMutation.isPending ? "Removendo..." : "Remover aula"}
              </Button>
            </DialogFooter>
          </DialogContent>
        </Dialog>

        </TabsContent>

        <TabsContent value="exams">
          <div className="rounded-xl border border-dashed p-8 text-center text-muted-foreground">
            Área de provas disponível em breve.
          </div>
        </TabsContent>

        <TabsContent value="certificate">
          <div className="rounded-xl border border-dashed p-8 text-center text-muted-foreground">
            Configuração de certificado disponível em breve.
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}
