// Student Header Component - Netflix-style navigation
import { LogOut, Menu, X } from "lucide-react";
import { Link } from "wouter";
import { useAuth } from "@/hooks/useAuth";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Button } from "@/components/ui/button";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetTitle,
  SheetTrigger,
} from "@/components/ui/sheet";
import { ThemeToggle } from "@/components/theme-toggle";
import { useState, useEffect } from "react";

export function StudentHeader() {
  const { user } = useAuth();
  const [isScrolled, setIsScrolled] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);

  useEffect(() => {
    const handleScroll = () => {
      setIsScrolled(window.scrollY > 10);
    };
    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, []);

  return (
    <header
      className="fixed top-0 left-0 right-0 z-50 transition-all duration-300 bg-background border-b border-border"
    >
      <div className="max-w-screen-2xl mx-auto px-4 sm:px-6 py-3 sm:py-4">
        <div className="flex items-center justify-between gap-4">
          {/* Logo - Same as Admin */}
          <Link href="/" data-testid="link-home">
            <div className="flex items-center gap-2 cursor-pointer">
              <span className="text-primary font-semibold text-lg sm:text-xl">Saber</span>
              <div className="bg-muted rounded-md px-2 sm:px-2.5 py-1 flex items-center gap-1.5">
                <span className="text-primary text-xs font-medium">ON</span>
                <div className="w-2 h-2 rounded-full bg-primary"></div>
              </div>
            </div>
          </Link>

          {/* Navigation Links - Desktop */}
          <nav className="flex-1 flex items-center justify-center gap-4 lg:gap-6 hidden md:flex">
            <Link href="/" className="text-sm lg:text-base text-foreground hover:text-primary transition-colors whitespace-nowrap">
              Início
            </Link>
            <a 
              href="https://cursos.saberon.com.br/course/92d9a1ee-58d8-43f8-95a6-e7f78511b5b4" 
              target="_blank" 
              rel="noopener noreferrer"
              className="text-sm lg:text-base text-foreground hover:text-primary transition-colors whitespace-nowrap"
            >
              Tour pela Plataforma
            </a>
            <Link href="/faq" className="text-sm lg:text-base text-foreground hover:text-primary transition-colors whitespace-nowrap">
              Perguntas Frequentes
            </Link>
            <Link href="/suporte" className="text-sm lg:text-base text-foreground hover:text-primary transition-colors whitespace-nowrap">
              Suporte
            </Link>
          </nav>

          {/* Right Side - Theme Toggle + Mobile Menu + User Menu */}
          <div className="flex items-center gap-2 sm:gap-3">
            {/* Theme Toggle */}
            <ThemeToggle />
            
            {/* Mobile Menu Button */}
            <Sheet open={mobileMenuOpen} onOpenChange={setMobileMenuOpen}>
              <SheetTrigger asChild>
                <Button variant="ghost" size="icon" className="md:hidden">
                  <Menu className="h-5 w-5" />
                </Button>
              </SheetTrigger>
              <SheetContent side="right" className="w-[300px] sm:w-[400px]">
                <SheetHeader>
                  <SheetTitle>Menu</SheetTitle>
                </SheetHeader>
                <nav className="flex flex-col gap-4 mt-6">
                  <Link 
                    href="/" 
                    onClick={() => setMobileMenuOpen(false)}
                    className="text-foreground hover:text-primary transition-colors py-2 text-lg"
                  >
                    Início
                  </Link>
                  <a 
                    href="https://cursos.saberon.com.br/course/92d9a1ee-58d8-43f8-95a6-e7f78511b5b4" 
                    target="_blank" 
                    rel="noopener noreferrer"
                    onClick={() => setMobileMenuOpen(false)}
                    className="text-foreground hover:text-primary transition-colors py-2 text-lg"
                  >
                    Tour pela Plataforma
                  </a>
                  <Link 
                    href="/faq" 
                    onClick={() => setMobileMenuOpen(false)}
                    className="text-foreground hover:text-primary transition-colors py-2 text-lg"
                  >
                    Perguntas Frequentes
                  </Link>
                  <Link 
                    href="/suporte" 
                    onClick={() => setMobileMenuOpen(false)}
                    className="text-foreground hover:text-primary transition-colors py-2 text-lg"
                  >
                    Suporte
                  </Link>
                  <div className="pt-4 border-t">
                    <div className="flex items-center justify-between py-2">
                      <span className="text-foreground">Tema</span>
                      <ThemeToggle />
                    </div>
                  </div>
                  <div className="pt-4 border-t">
                    <div className="flex items-center gap-3 mb-4">
                      <Avatar className="h-10 w-10">
                        <AvatarImage src={user?.profileImageUrl || ""} />
                        <AvatarFallback>
                          {user?.firstName?.[0]}{user?.lastName?.[0]}
                        </AvatarFallback>
                      </Avatar>
                      <div>
                        <p className="font-medium text-sm">{user?.firstName} {user?.lastName}</p>
                        <p className="text-xs text-muted-foreground truncate">{user?.email}</p>
                      </div>
                    </div>
                    <Button
                      variant="outline"
                      className="w-full justify-start"
                      onClick={() => {
                        setMobileMenuOpen(false);
                        window.location.href = "/api/logout-redirect";
                      }}
                    >
                      <LogOut className="w-4 h-4 mr-2" />
                      Sair
                    </Button>
                  </div>
                </nav>
              </SheetContent>
            </Sheet>

            {/* User Menu - Desktop */}
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <Button variant="ghost" className="relative h-9 w-9 sm:h-10 sm:w-10 rounded-full" data-testid="button-user-menu">
                  <Avatar className="h-9 w-9 sm:h-10 sm:w-10">
                    <AvatarImage src={user?.profileImageUrl || ""} />
                    <AvatarFallback className="text-xs">
                      {user?.firstName?.[0]}{user?.lastName?.[0]}
                    </AvatarFallback>
                  </Avatar>
                </Button>
              </DropdownMenuTrigger>
              <DropdownMenuContent align="end" className="w-56">
                <DropdownMenuLabel>
                  <div>
                    <p className="font-medium">{user?.firstName} {user?.lastName}</p>
                    <p className="text-xs text-muted-foreground truncate">{user?.email}</p>
                  </div>
                </DropdownMenuLabel>
                <DropdownMenuSeparator />
                <DropdownMenuItem onClick={() => window.location.href = "/api/logout-redirect"} data-testid="menu-logout">
                  <LogOut className="w-4 h-4 mr-2" />
                  Sair
                </DropdownMenuItem>
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
        </div>
      </div>
    </header>
  );
}
