// Admin Sidebar Component - Following Shadcn sidebar patterns and design guidelines
import {
  LayoutDashboard,
  Store,
  Users,
  DollarSign,
  HeadphonesIcon,
  LogOut,
  GraduationCap,
} from "lucide-react";
import { Link, useLocation } from "wouter";
import {
  Sidebar,
  SidebarContent,
  SidebarGroup,
  SidebarGroupContent,
  SidebarGroupLabel,
  SidebarMenu,
  SidebarMenuButton,
  SidebarMenuItem,
  SidebarHeader,
  SidebarFooter,
} from "@/components/ui/sidebar";
import { useAuth } from "@/hooks/useAuth";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Button } from "@/components/ui/button";
import { ThemeToggle } from "@/components/theme-toggle";
import logoUrl from "@assets/logo.png";

const adminMenuItems = [
  {
    title: "Dashboard",
    url: "/admin",
    icon: LayoutDashboard,
  },
  {
    title: "Vitrine",
    url: "/admin/vitrine",
    icon: Store,
  },
  {
    title: "Usuários",
    url: "/admin/users",
    icon: Users,
  },
  {
    title: "Vendas",
    url: "/admin/sales",
    icon: DollarSign,
  },
  {
    title: "Atendimento",
    url: "/admin/support",
    icon: HeadphonesIcon,
  },
  {
    title: "Área do aluno",
    url: "/admin/student-view",
    icon: GraduationCap,
  },
];

export function AppSidebar() {
  const [location] = useLocation();
  const { user } = useAuth();
  const toggleTheme = () => {
    const root = document.documentElement;
    const isDark = root.classList.contains("dark");
    if (isDark) {
      root.classList.remove("dark");
      localStorage.setItem("theme", "light");
    } else {
      root.classList.add("dark");
      localStorage.setItem("theme", "dark");
    }
  };

  return (
    <Sidebar>
      <SidebarHeader className="p-6">
        <div className="flex items-center gap-3">
          <img 
            src={logoUrl} 
            alt="Portal de Cursos" 
            className="h-10 w-auto object-contain"
          />
          <div>
            <h2 className="font-semibold text-lg font-serif">Portal</h2>
            <p className="text-xs text-muted-foreground">Área Admin</p>
          </div>
        </div>
      </SidebarHeader>
      
      <SidebarContent>
        {/* Quick actions to ensure visibility */}
        <SidebarGroup>
          <SidebarGroupLabel>Acesso rápido</SidebarGroupLabel>
          <SidebarGroupContent>
            <SidebarMenu>
              <SidebarMenuItem>
                <SidebarMenuButton asChild isActive={location === "/admin/student-view"}>
                  <Link href="/admin/student-view" data-testid="link-area-aluno">
                    <GraduationCap className="w-4 h-4" />
                    <span>Área do aluno</span>
                  </Link>
                </SidebarMenuButton>
              </SidebarMenuItem>
              <SidebarMenuItem>
                <SidebarMenuButton onClick={toggleTheme} data-testid="link-alternar-tema">
                  <span className="w-4 h-4 rounded-sm bg-muted inline-block" />
                  <span>Alternar tema</span>
                </SidebarMenuButton>
              </SidebarMenuItem>
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>

        <SidebarGroup>
          <SidebarGroupLabel>Navegação</SidebarGroupLabel>
          <SidebarGroupContent>
            <SidebarMenu>
              {adminMenuItems.map((item) => (
                <SidebarMenuItem key={item.title}>
                  <SidebarMenuButton asChild isActive={location === item.url}>
                    <Link href={item.url} data-testid={`link-${item.title.toLowerCase()}`}>
                      <item.icon className="w-4 h-4" />
                      <span>{item.title}</span>
                    </Link>
                  </SidebarMenuButton>
                </SidebarMenuItem>
              ))}
              <SidebarMenuItem>
                <SidebarMenuButton onClick={toggleTheme} data-testid="toggle-tema">
                  <span className="w-4 h-4 inline-block rounded-sm bg-muted" />
                  <span>Alternar tema</span>
                </SidebarMenuButton>
              </SidebarMenuItem>
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>

      <SidebarFooter className="p-4">
        <div className="flex items-center justify-between mb-2">
          <span className="text-xs text-muted-foreground">Tema</span>
          <ThemeToggle />
        </div>
        <div className="flex items-center gap-3 p-2 rounded-md hover-elevate">
          <Avatar className="w-8 h-8">
            <AvatarImage src={user?.profileImageUrl || ""} />
            <AvatarFallback className="text-xs">
              {user?.firstName?.[0]}{user?.lastName?.[0]}
            </AvatarFallback>
          </Avatar>
          <div className="flex-1 overflow-hidden">
            <p className="text-sm font-medium truncate">
              {user?.firstName} {user?.lastName}
            </p>
            <p className="text-xs text-muted-foreground truncate">{user?.email}</p>
          </div>
        </div>
        <Button
          variant="ghost"
          className="w-full justify-start mt-2"
          onClick={() => window.location.href = "/api/logout-redirect"}
          data-testid="button-logout"
        >
          <LogOut className="w-4 h-4 mr-2" />
          Sair
        </Button>
      </SidebarFooter>
    </Sidebar>
  );
}
