// Admin Header Component - Horizontal navigation menu
import {
  Settings,
  ChevronDown,
  LogOut,
  Menu,
} from "lucide-react";
import { Link, useLocation } from "wouter";
import { useAuth } from "@/hooks/useAuth";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Button } from "@/components/ui/button";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetTitle,
  SheetTrigger,
} from "@/components/ui/sheet";
import { ThemeToggle } from "@/components/theme-toggle";
import { cn } from "@/lib/utils";
import { useState } from "react";

const adminMenuItems = [
  {
    title: "Dashboard",
    url: "/admin",
  },
  {
    title: "Vitrine",
    url: "/admin/vitrine",
  },
  {
    title: "Usuários",
    url: "/admin/users",
  },
  {
    title: "Vendas",
    url: "/admin/sales",
  },
  {
    title: "Atendimento",
    url: "/admin/support",
  },
];

export function AdminHeader() {
  const [location] = useLocation();
  const { user } = useAuth();
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);

  // Normalize location for active state checking
  const currentPath = location === "/" || location === "/admin" || location === "/admin/dashboard" 
    ? "/admin" 
    : location;

  return (
    <header className="sticky top-0 z-50 w-full border-b bg-background">
      <div className="container flex h-14 sm:h-16 items-center px-3 sm:px-4">
        {/* Logo - Left */}
        <Link href="/admin" className="flex items-center gap-2 mr-4 sm:mr-8 shrink-0">
          <div className="flex items-center gap-1.5 sm:gap-2">
            <span className="text-[#60A5FA] font-semibold text-lg sm:text-xl">Saber</span>
            <div className="bg-[#374151] rounded-md px-2 sm:px-2.5 py-1 flex items-center gap-1.5">
              <span className="text-[#60A5FA] text-xs font-medium">ON</span>
              <div className="w-2 h-2 rounded-full bg-[#60A5FA]"></div>
            </div>
          </div>
        </Link>

        {/* Navigation Menu - Center (Desktop) */}
        <nav className="hidden lg:flex items-center gap-1 flex-1 justify-center">
          {adminMenuItems.map((item) => {
            const isActive = currentPath === item.url;
            return (
              <Link
                key={item.url}
                href={item.url}
                className={cn(
                  "px-3 xl:px-4 py-2 rounded-md text-xs xl:text-sm font-medium transition-colors whitespace-nowrap",
                  isActive
                    ? "text-[#EC4899] bg-transparent"
                    : "text-[#6B7280] hover:text-foreground hover:bg-accent"
                )}
              >
                {item.title}
              </Link>
            );
          })}
          <Button variant="ghost" size="sm" className="ml-2 text-[#6B7280] hover:text-foreground">
            <Settings className="w-4 h-4" />
          </Button>
        </nav>

        {/* Right Side - Mobile Menu + Theme Toggle + User Menu */}
        <div className="flex items-center gap-2 sm:gap-3 ml-auto">
          {/* Mobile Menu Button */}
          <Sheet open={mobileMenuOpen} onOpenChange={setMobileMenuOpen}>
            <SheetTrigger asChild>
              <Button variant="ghost" size="icon" className="lg:hidden">
                <Menu className="h-5 w-5" />
              </Button>
            </SheetTrigger>
            <SheetContent side="right" className="w-[300px] sm:w-[400px]">
              <SheetHeader>
                <SheetTitle>Menu de Navegação</SheetTitle>
              </SheetHeader>
              <nav className="flex flex-col gap-2 mt-6">
                {adminMenuItems.map((item) => {
                  const isActive = currentPath === item.url;
                  return (
                    <Link
                      key={item.url}
                      href={item.url}
                      onClick={() => setMobileMenuOpen(false)}
                      className={cn(
                        "px-4 py-3 rounded-md text-base font-medium transition-colors",
                        isActive
                          ? "text-[#EC4899] bg-accent"
                          : "text-foreground hover:text-[#EC4899] hover:bg-accent"
                      )}
                    >
                      {item.title}
                    </Link>
                  );
                })}
                <div className="pt-4 border-t mt-4">
                  <div className="flex items-center gap-3 mb-4">
                    <Avatar className="h-10 w-10">
                      <AvatarImage src={user?.profileImageUrl || ""} />
                      <AvatarFallback>
                        {user?.firstName?.[0]}{user?.lastName?.[0]}
                      </AvatarFallback>
                    </Avatar>
                    <div>
                      <p className="font-medium text-sm">{user?.firstName} {user?.lastName}</p>
                      <p className="text-xs text-muted-foreground truncate">{user?.email}</p>
                    </div>
                  </div>
                  <Button
                    variant="outline"
                    className="w-full justify-start mb-2"
                    onClick={() => {
                      setMobileMenuOpen(false);
                      window.location.href = "/admin/student-view";
                    }}
                  >
                    Área do aluno
                  </Button>
                  <Button
                    variant="outline"
                    className="w-full justify-start"
                    onClick={() => {
                      setMobileMenuOpen(false);
                      window.location.href = "/api/logout-redirect";
                    }}
                  >
                    <LogOut className="w-4 h-4 mr-2" />
                    Sair
                  </Button>
                </div>
              </nav>
            </SheetContent>
          </Sheet>

          {/* Theme Toggle */}
          <ThemeToggle />

          {/* User Menu - Desktop */}
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <Button
                variant="ghost"
                className="hidden lg:flex items-center gap-2 h-9 px-2 hover:bg-accent"
              >
                <Avatar className="h-7 w-7">
                  <AvatarImage src={user?.profileImageUrl || ""} />
                  <AvatarFallback className="text-xs">
                    {user?.firstName?.[0]}{user?.lastName?.[0]}
                  </AvatarFallback>
                </Avatar>
                <ChevronDown className="w-4 h-4 text-muted-foreground" />
              </Button>
            </DropdownMenuTrigger>
            <DropdownMenuContent align="end" className="w-56">
              <DropdownMenuLabel>
                <div>
                  <p className="font-medium">{user?.firstName} {user?.lastName}</p>
                  <p className="text-xs text-muted-foreground truncate">{user?.email}</p>
                </div>
              </DropdownMenuLabel>
              <DropdownMenuSeparator />
              <DropdownMenuItem onClick={() => window.location.href = "/admin/student-view"}>
                Área do aluno
              </DropdownMenuItem>
              <DropdownMenuItem onClick={() => window.location.href = "/api/logout-redirect"}>
                <LogOut className="w-4 h-4 mr-2" />
                Sair
              </DropdownMenuItem>
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
      </div>
    </header>
  );
}
