// Main App - Routing and Auth setup following Replit Auth blueprint
import { Switch, Route, useLocation } from "wouter";
import { queryClient } from "./lib/queryClient";
import { QueryClientProvider } from "@tanstack/react-query";
import { Toaster } from "@/components/ui/toaster";
import { TooltipProvider } from "@/components/ui/tooltip";
import { AdminHeader } from "@/components/admin-header";
import { useAuth } from "@/hooks/useAuth";
import NotFound from "@/pages/not-found";
import Landing from "@/pages/landing";
import AdminDashboard from "@/pages/admin-dashboard";
import AdminVitrine from "@/pages/admin-vitrine";
import AdminCourseContent from "@/pages/admin-course-content";
import AdminLessonNew from "@/pages/admin-lesson-new";
import AdminUsers from "@/pages/admin-users";
import AdminSales from "@/pages/admin-sales";
import AdminSupport from "@/pages/admin-support";
import StudentDashboard from "@/pages/student-dashboard";
import StudentFAQ from "@/pages/student-faq";
import StudentSupport from "@/pages/student-support";
import CoursePlayer from "@/pages/course-player";

function Router() {
  const { isAuthenticated, isLoading, isAdmin } = useAuth();
  const [location] = useLocation();

  // Show loading spinner while checking authentication
  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-background">
        <div className="flex flex-col items-center space-y-4">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          <p className="text-sm text-muted-foreground">Carregando...</p>
        </div>
      </div>
    );
  }

  // Show landing page for unauthenticated users
  if (!isAuthenticated) {
    return <Landing />;
  }

  // Admin routes com header horizontal
  if (isAdmin) {
    // Special case: admin visualizando como aluno em tela cheia (sem layout do admin)
    // Permite acesso às rotas de estudante quando visualizando como aluno
    if (location === "/admin/student-view" || location.startsWith("/course/") || location === "/faq" || location === "/suporte") {
      return (
        <Switch>
          <Route path="/faq" component={StudentFAQ} />
          <Route path="/suporte" component={StudentSupport} />
          <Route path="/admin/student-view" component={StudentDashboard} />
          <Route path="/course/:id">
            {(params) => <CoursePlayer params={params} />}
          </Route>
          <Route component={NotFound} />
        </Switch>
      );
    }
    return (
      <div className="flex flex-col h-screen w-full">
        <AdminHeader />
        <main className="flex-1 overflow-auto p-3 sm:p-4 lg:p-6 bg-background">
          <div className="max-w-7xl mx-auto">
            <Switch>
              <Route path="/" component={AdminDashboard} />
              <Route path="/admin" component={AdminDashboard} />
              <Route path="/admin/dashboard" component={AdminDashboard} />
              {/* Rotas do admin */}
              <Route path="/admin/vitrine" component={AdminVitrine} />
              <Route path="/admin/courses/:courseId/content" component={AdminCourseContent} />
              <Route
                path="/admin/courses/:courseId/modules/:moduleId/lessons/new"
                component={AdminLessonNew}
              />
              <Route path="/admin/users" component={AdminUsers} />
              <Route path="/admin/sales" component={AdminSales} />
              <Route path="/admin/support" component={AdminSupport} />
              <Route component={NotFound} />
            </Switch>
          </div>
        </main>
      </div>
    );
  }

  // Student routes without sidebar
  return (
    <Switch>
      <Route path="/faq" component={StudentFAQ} />
      <Route path="/suporte" component={StudentSupport} />
      <Route path="/course/:id">
        {(params) => <CoursePlayer params={params} />}
      </Route>
      <Route path="/" component={StudentDashboard} />
      <Route component={NotFound} />
    </Switch>
  );
}

export default function App() {
  return (
    <QueryClientProvider client={queryClient}>
      <TooltipProvider>
        <Toaster />
        <Router />
      </TooltipProvider>
    </QueryClientProvider>
  );
}
